/******************************************************************************/
/* psx-sci.inl                            Standard Commandline Interface      */
/******************************************************************************/
/** @file psx-sci.inl Standard Commandline Interface - Inline Source Code File
 * Functions and definitions supporting the commandline option management like
 * the definition of the option type structure and target structure.
 */


//#define SCI_OPT_ITERATE(i,s)	\
//	for (i = 1;i < sci.a -> c;i++) if ((s = sci.a -> v [i])[0] == '-')
#define SCI_OPT_ITERATE(i,s) for (i = 1; i < sci.a->c; i++) if ((s = sci.a->v[i])[0] == '-')

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define SCI_ITP_ZERO	0x00
#define SCI_ITP_TGT	0x01
#define SCI_ITP_SYN	0x02
#define SCI_ITP_DEF	0x03

#define SCI_TTP_ZERO	0x00
#define SCI_TTP_PRC	0x01
#define SCI_TTP_INT	0x02

typedef struct _SCI_TGT
{
 BYTE    ttp;
 SCI_PRC prc;
 SCI_INT * itf;
} SCI_TGT;

static struct
{
 OTP otp;
 const char * sym;
} opt_otp [] =
{
 { OPT_BOOL,"b" },
 { OPT_INT ,"i" },
 { OPT_NUM ,"n" },
 { OPT_STR ,"s" },
 { OPT_ZERO,0   }
};

#define SCI_ITF_END(itf) ((itf).sym == NULL && (itf).ptr == NULL && (itf).opt == NULL && (itf).syn == NULL)

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get option type from string sym
 * @param sym symbolic short name of option type
 * @param otp option type structure, contains numeric code for option type after invocation
 */

static BOOL sci_opt_type (SYS_STR sym,OTP * otp)
{  
 IDX i;

 if (sym == NULL)
  return (FALSE);

 for (i = 0;opt_otp [i].sym != NULL;i++)    // for every option type element
  if (strcmp (opt_otp [i].sym,sym) == 0)    // look for option type sym
  {
   if (otp)                                 
    *otp = opt_otp [i].otp;

   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/** verify and store option, convert if necessary
 * @param otp option type structure
 * @param v value string
 * @param p pointer to result (string or number)
 */


static BOOL sci_opt_store (OTP otp,const char * v,PTR p)
{
 if (p == NULL)
  return (FALSE);



 switch (otp)
 {
  case OPT_ZERO:           // do same as for next case
  case OPT_BOOL:
  {
   BOOL t = FALSE;

   if (v)
    if (strcmp (v,"") == 0 || strcmp (v,"1") == 0 || strcmp (v,"true") == 0)
     t = TRUE;

   *((BOOL*) p) = t;
   break;
  }

  case OPT_INT:
  {
   *((int*) p) = v ? atoi (v) : 0;           // v is a number: typecast
   break;
  }

  case OPT_NUM:
  {
   *((NUM*) p) = v ? atoi (v) : 0;           // v is a number: typecast
   break;
  }

  case OPT_STR:                              // v is string
  {
   if (v)
   {
    int l = strlen (v);
    char * t = (char*) p;

    if (l < SYS_STR_MAX)
     strcpy (t,v);                          // copy v to t
    else                             
    {
     strncpy (t,v,SYS_STR_MAX - 1);         // avoid buffer overflow
     t [SYS_STR_MAX] = 0;
    }
   }
   else
    strcpy ((char*)p,"");                   // v empty
   break;
  }

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/** Split at ':', first part stored in a (attribute), second part in v (value)
 * @param o option string
 * @param a attribute string
 * @param v value string
 */

static BOOL sci_opt_split (char * o,char * a,char * v)
{
 char * p;
 int k;
 int n;

 if (o == NULL || a == NULL || v == NULL)
  return (FALSE);

 if ((p = strchr (o,':')) == NULL)
 {
  /* added 16.09.2004: avoid buffer overflow if o > a, o > v                */
  /* a, v: SYS_STR                                                          */
  int n = strlen (o);
  
  if (n > SYS_STR_MAX)
    n = SYS_STR_MAX;
  
  strncpy (a,o,n);
  a [n] = '\0';
      
  strcpy (v,"");

  return (TRUE);
 }

 k = p - o;
 
 if ( k > SYS_STR_MAX)  /* added 16.09.2004                                    */
   k = SYS_STR_MAX;
   
 strncpy (a,o,k);
 a [k] = '\0';

 n = strlen(p + 1);
 
 if (n > SYS_STR_MAX)   /* added 16.09.2004                                    */
  n = SYS_STR_MAX;
  
 strncpy (v,p + 1,n);
 v [n] = '\0';

 return (TRUE);
}

/******************************************************************************/
/** find specified command line option "-sym" and store obtained value in val
 * @param sym option string
 * @param val obtained value string
 */

static BOOL sci_opt_find (char * sym,SYS_STR val)
{
 IDX    i;
 char * o;

 if (sym == NULL || val == NULL)
  return (FALSE);
  
 /* iterates all supplied command line options starting with "-"              */
 /* i = index of option, o = option                                           */
 SCI_OPT_ITERATE (i,o)   /* defined at line 12                                */
 {
  SYS_STR s,v;

  sci_opt_split (o + 1,s,v);    /*  split string (without "-")                */

  if (strcmp (s,sym) != 0)      /*  if optionname != sym -> continue search   */
   continue;

  strcpy (val,v);
  return (TRUE);
 }

 return (FALSE);
}

/******************************************************************************/
/** parse options, store in sym
 * @param o option string
 * @param sym retrieved option string
 * @param otp option type
 */

static BOOL sci_opt_int_parse (SYS_STR o,SYS_STR sym,OTP * otp)

{
 SYS_STR s,t;

 if (o == NULL || sym == NULL || otp == NULL)
  return (FALSE);

 if (!sci_opt_split (o,s,t))           // split string, retrieve option
  return (FALSE);

 strcpy (sym,s);

 if (t [0] != 0)
 {
  if (!sci_opt_type (t,otp))           // get option type
   return (FALSE);
 }
 else
  *otp = OPT_BOOL;

 return (TRUE);
}

/******************************************************************************/
/* parse options                                                              */

/* 16.09.2004: commented out, not needed
static BOOL sci_opt_parse_put (SCI_INT_OPT * o,SYS_STR s,SYS_STR v)
{
 IDX i;

 if (o == NULL || s == NULL)
  return (FALSE);

 for (i = 0;o [i].sym != NULL;i++)
 {
  if (strcmp (s,o [i].sym) != 0)
   continue;


//  if (!sci_opt_put (&o [i],v))
//   return (FALSE); 

  return (TRUE);
 }

 return (FALSE);
}
*/
/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/******************************************************************************/
/** retrieve parameters from sci.a, store in s
 * @param s string opinter to retrieved option parameter after invocation
 * @param idx number of parameters after invocation
 */
 
static BOOL sci_par_retrieve (char * s,IDX * idx)
{
 IDX i;

 assert (sci.a != NULL);	// argv[] stored in sci.a

 for (i = sci.p + 1;i < sci.a -> c;i++)
 {
  if (sci.a -> v [i][0] == '-')
   continue;

  if (s)
  {
    int n;    /* added 16.09.2004                                             */
    if ((n = strlen (sci.a -> v[i])) > PSX_STR_MAX)
      n = PSX_STR_MAX;
    strncpy (s,sci.a -> v [i],n);        // store option parameter in s 
    s [n] = '\0';
  }
  if (idx)
   *idx = i;

  return (TRUE);
 }

 return (FALSE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/** test if str begins with pfx
 * @param str string to be checked
 * @param pfx specified prefix string
 */

static BOOL sci_str_pfx (char * str,char * pfx)
{
 if (str == NULL || pfx == NULL)
  return (FALSE);

 if (strncmp (str,pfx,strlen (pfx)) == 0)
  return (TRUE);

 return (FALSE);
}

/******************************************************************************/
/** get idxth element of interface itf, increment idx to point to next element
 * @param itf interface
 * @param idx index
 * @param itp information type (BYTE representation)
 */

static BOOL sci_itf_get (SCI_INT * itf,IDX * idx,BYTE itp)
{
 IDX i;

 if (itf == NULL || idx == NULL)

  return (FALSE);

 switch (itp)
 {
  case SCI_ITP_TGT:	//  0x01
  {
   for (i = *idx;!SCI_ITF_END (itf [i]);i++) // ((itf).sym == NULL && (itf).ptr == NULL
    if (itf [i].sym != NULL)                 //&& (itf).opt == NULL && (itf).syn == NULL)
    {
     *idx = i;                               // pointer idx now points to next element
     return (TRUE);
    }

   break;
  }

  case SCI_ITP_SYN:	//  0x02
  {
  // iterates as long as elements of itf contain values
   for (i = 0;itf [i].sym != NULL || itf [i].ptr != NULL || itf [i].syn == NULL;i++)
    ;
   if (itf [i].syn == NULL)
    return (FALSE);

   *idx = i;        // idx now points to last element (= "hlp")

   return (TRUE);
  }

  case SCI_ITP_DEF: //  0x03
  {
   for (i = 0;!SCI_ITF_END (itf [i]);i++)
    if (itf [i].sym == NULL && itf [i].ptr != NULL)   // sym does not exist but ptr does
     break;                                           // = return(FALSE)

   if (!(itf [i].sym == NULL && itf [i].ptr != NULL)) // both do not exist
    return (FALSE);

   assert (itf [i].sym == NULL);                      // make sure

   if (itf [i].ptr == NULL)
    return (FALSE);

   *idx = i;       // pointer to index of last element

   return (TRUE);
  }

  default:
   return (FALSE);
 }


 return (FALSE);
}

/******************************************************************************/
/** check if itf[idx].sym closes with '+'
 * @param itf sci interface structure
 * @param idx index
 */
static BOOL sci_tgt_isint (SCI_INT * itf,IDX idx)
{
 int l,k;
 char * s;

 if (itf == NULL)
  return (FALSE);

 if (itf [idx].sym == NULL)
  return (FALSE);

 s = itf [idx].sym;

 if ((l = strlen (s)) == 0)
  return (FALSE);

 if (s [l - 1] != '+')           // '+' at last position
  return (FALSE);

 return (TRUE);
}


/******************************************************************************/
/** get symbol from str and store in sym; check whether last character of str is '+'
 * @param str input string
 * @param sym result string
 * @param isitf whether last character is '+'
 */

static BOOL sci_tgt_get_sym (const SYS_STR str,SYS_STR sym,BOOL * isitf)
{
 int l,k;

 if ((l = strlen (str)) == 0)
 {
  strcpy (sym,"");

  if (isitf)
   *isitf = FALSE;

  return (FALSE);              // no string found
 }

 k = l - 1;

 if (str [k] == '+')           // last character '+' 
 {
  strncpy (sym,str,k);
  sym [k] = 0;

  if (isitf)
   *isitf = TRUE;
 }
 else
 {
  strcpy (sym,str);           // store str in sym 

  if (isitf)
   *isitf = FALSE;
 }

 return (TRUE);
}

/******************************************************************************/
/** fill in values to target structure
 * @param itf sci interface structure
 * @param idx index
 * @param tgt sci target structure
 */

static BOOL sci_tgt_fetch (SCI_INT * itf,IDX idx,SCI_TGT * tgt)
{
 if (itf == NULL || tgt == NULL)
  return (FALSE);

 memset (tgt,0,sizeof (*tgt));


 //if (idx >= itf -> n)
 // return (FALSE);

 if (sci_tgt_isint (itf,idx))      // check if it is really an interface    
 {
  tgt -> ttp = SCI_TTP_INT; // =  0x02
  tgt -> itf = itf [idx].ptr;
 }
 else
 {
  tgt -> ttp = SCI_TTP_PRC;         // no interface, but procedure
  tgt -> prc = itf [idx].ptr;

 }

 return (TRUE);
}

/******************************************************************************/
/** stores index of command sym in idx, checks whether corresponding procedure exists
 * (lookup informationtype definition)
 * @param itf (command line) interface
 * @param sym name of command
 * @param idx index of command
 * @param pfx whether command name is set (???)
 */

static BOOL sci_tgt_lookup (SCI_INT * itf,char * sym,IDX * idx,BOOL pfx)
{
 IDX i,j;

 if (itf == NULL || idx == NULL)
  return (FALSE);

 /*************************************/

 if (sym == NULL)   // no command name
 {
  i = 0;

  if (!sci_itf_get (itf,&i,SCI_ITP_DEF))   // SCI_ITP_DEF = 0x03

   return (FALSE);

  *idx = i;

  return (TRUE);
 }

 /*************************************/

 assert (sym != NULL);

 /*************************************/
 // sci_itf_get: get pointer to ith element itf
 for (i = 0;sci_itf_get (itf,&i,SCI_ITP_TGT);i++)  // SCI_ITP_TGT = 0x01
 {
  SYS_STR t;

  sci_tgt_get_sym (itf [i].sym,t,NULL);  // store name of ith command in t ("hlp", "req", "ver"...)

  if (strcmp (t,sym) == 0) // 
  {
   *idx = i;                             // store index of command in idx

   return (itf [i].ptr != NULL);
  }
 }

 /*************************************/

 if (!pfx)
  return (FALSE);

 //  if (!sci_tgt_get_pfx (itf,sym,idx))
 //   return (FALSE);

 /*************************************/

 i = 0;
 j = IDX_NIL;

 for (i = 0;sci_itf_get (itf,&i,SCI_ITP_TGT);i++)
 {
  SYS_STR t;

  sci_tgt_get_sym (itf [i].sym,t,NULL); // itf[i].sym is stored in t, '+' gets cut off
  if (sci_str_pfx (t,sym))              // check if t starts with sym
   if (j == IDX_NIL)
    j = i;             // j is only set once, index of first equality
   else
    return (FALSE);
 }

 if (j == IDX_NIL)
  return (FALSE);

 if (itf [j].ptr == NULL)
  return (FALSE);

 *idx = j;             // index of command name

 return (TRUE);
}

/******************************************************************************/
/** get target structure (target type and target procedure)
 * @param itf pointer to (command line) interface
 * @param tgt target pointer
 */

static BOOL sci_tgt_get (SCI_INT * itf,SCI_TGT * tgt)
{

 SYS_STR par;
 IDX idx;

 idx = IDX_NIL;

 if (sci_par_peek (par))  //get command name
 {
  if (sci_tgt_lookup (itf,par,&idx,TRUE)) //stores index of command sym in idx, checks whether corresponding procedure exists
   sci_par_fetch (NULL);
  else
   if (!sci_tgt_lookup (itf,NULL,&idx,FALSE))
    return (FALSE);
 }
 else
 {
  if (!sci_tgt_lookup (itf,NULL,&idx,FALSE))
   return (FALSE);
 }

 if (idx == IDX_NIL)
  return (FALSE);                             // no target found, no symbol existent

 //if (idx > itf -> n)
 // return (FALSE);

 if (!sci_tgt_fetch (itf,idx,tgt))
  return (FALSE);

 return (TRUE);
}



/******************************************************************************/
/******************************************************************************/

/******************************************************************************/





