/******************************************************************************/
/* tcc-psr.y                                                                  */
/******************************************************************************/

%{

/******************************************************************************/
/* Interface                                                                  */
/******************************************************************************/

#include "psx.h"

/**************************************/
/* Crash Configuration                */
/**************************************/

#define YYSTACK_USE_ALLOCA 0
#define YYINITDEPTH 20

/**************************************/
/* Interface Definitions              */
/**************************************/

typedef struct
{
 CCI_TOKEN   tok;
 FSP_ITM     fld;
 RSP_ITM     rsl;
 PSP_ITM     sts;
} SPC_PSR_TOKEN;

#define YYDEBUG 1
//#define YYSTYPE CCI_TOKEN
#define YYSTYPE SPC_PSR_TOKEN

static PSX_SPI * spi = NULL;

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

BOOL spi_psr_cfg (PSX_SPI * s)
{
 spi = s;

 return (TRUE);
}

/******************************************************************************/

static BOOL sch_fld_pass (FMT_ITM * f,SPC_PSR_TOKEN * t)
{
 if (f == NULL || t == NULL)
  return (FALSE);

 switch (t -> tok.idx)
 {
  case FSP_ITM_SYM:	strcpy (f -> sym,t -> tok.txt);	break;
  case FSP_ITM_DTP:	f -> dtp = t -> tok.num;	break;
  case FSP_ITM_LBL:	strcpy (f -> lbl,t -> tok.txt);	break;
  case FSP_ITM_POS:	f -> pos = t -> tok.num;	break;
  case FSP_ITM_LEN:	f -> len = t -> tok.num;	break;
  case FSP_ITM_MIN:	f -> min = t -> tok.num;	break;
  case FSP_ITM_MAX:	f -> max = t -> tok.num;	break;
  case FSP_ITM_TFM:	f -> tfm = t -> tok.num;	break;
  case FSP_ITM_EQU:	f -> equ = t -> tok.num;	break;
  case FSP_ITM_CTT:	f -> ctt = t -> tok.num;	break;

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/

static BOOL sch_rsl_pass (RSP_ITM * r,SPC_PSR_TOKEN * t)
{
 if (r == NULL || t == NULL)
  return (FALSE);

 switch (t -> tok.idx)
 {
  case RSP_ITM_PRM:	r -> prm = t -> tok.itg;	break;
  case RSP_ITM_RUM:	r -> rum = t -> tok.itg;	break;
  case RSP_ITM_MSG:	strcpy (r -> msg,t -> tok.txt);	break;
  case RSP_ITM_NOT:	strcpy (r -> not,t -> tok.txt); break;

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/

static BOOL sch_sts_pass (PSP_ITM * s,SPC_PSR_TOKEN * t)
{
 if (s == NULL || t == NULL)
  return (FALSE);

 switch (t -> tok.idx)
 {
  case PSP_ITM_STT:	s -> pro.stt = t -> tok.itg;	break;
  case PSP_ITM_MTM:	s -> pro.mtm |= t -> tok.itg;	break;
  case PSP_ITM_TGT_0:	strcpy (s -> epi.tgt_0.sym,t -> tok.txt);	break;
  case PSP_ITM_TGT_1:	strcpy (s -> epi.tgt_1.sym,t -> tok.txt);	break;
  case PSP_ITM_TGT_M:	strcpy (s -> epi.tgt_m.sym,t -> tok.txt);	break;

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/

static BOOL sch_sts_resolve_s (PSX_SPI * spi,IDX idx,PSX_STR sym,IDX * tgt)
{
 IDX i;

 *tgt = IDX_NIL;

 for (i = 0;i < spi -> psp -> n;i++)
 {
  if (strcmp (spi -> psp -> itm [i].sym,sym) == 0)
   if (*tgt == IDX_NIL)
   {
    *tgt = i;
   }
   else
   {
    cci_error ("multiple test targets '%s' for test '%s'",sym,spi -> psp -> itm [idx].sym);
    return (FALSE);
   }
 }

 if (*tgt == IDX_NIL)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/

static BOOL sch_sts_resolve_r (PSX_SPI * spi,IDX idx,PSX_STR sym,IDX * tgt)
{
 IDX i;

 *tgt = IDX_NIL;

 for (i = 0;i < spi -> rsp -> n;i++)
 {
  if (strcmp (spi -> rsp -> itm [i].sym,sym) == 0)
   if (*tgt == IDX_NIL)
   {
    *tgt = i;
   }
   else
   {
    cci_error ("multiple result targets '%s' for test '%s'",sym,spi -> psp -> itm [idx].sym);
    return (FALSE);
   }
 }

 if (*tgt == IDX_NIL)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* syn: resolve target */

static BOOL sch_sts_resolve_t (PSX_SPI * spi,IDX idx,PSP_TGT * tgt)
{
 BOOL rs,rr;
 IDX ts,tr;

 if (str_isnil (tgt -> sym))
 {
  cci_error ("empty target in test '%s'\n",spi -> psp -> itm [idx].sym);
  return (FALSE);
 }

 rs = sch_sts_resolve_s (spi,idx,tgt -> sym,&ts);
 rr = sch_sts_resolve_r (spi,idx,tgt -> sym,&tr);

 if (rs)
 {
  if (rr)
  {
   cci_error ("ambiguous target '%s' in test '%s'",tgt -> sym,spi -> psp -> itm [idx].sym);
   return (FALSE);
  }
  else
  {
   tgt -> sdt = SDT_STS;
   tgt -> idx = ts;
  }
 }
 else
 {
  if (rr)
  {
   tgt -> sdt = SDT_RSL;
   tgt -> idx = tr;
  }
  else
  {
   cci_error ("unresolved target '%s' in test '%s'",tgt -> sym,spi -> psp -> itm [idx].sym);
   return (FALSE);
  }
 }

 return (TRUE);
}

/******************************************************************************/

static BOOL sch_sts_resolve (PSX_SPI * spi)
{
 IDX i,j;

 for (i = 0;i < spi -> psp -> n;i++)
 {
  if (!sch_sts_resolve_t (spi,i,&spi -> psp -> itm [i].epi.tgt_0))
  {
   cci_error ("unresolved target '%s' in test '%s'",spi -> psp -> itm [i].epi.tgt_0.sym,spi -> psp -> itm [i].sym);
   return (FALSE);
  }

  if (!sch_sts_resolve_t (spi,i,&spi -> psp -> itm [i].epi.tgt_1))
  {
   cci_error ("unresolved target '%s' in test '%s'",spi -> psp -> itm [i].epi.tgt_1.sym,spi -> psp -> itm [i].sym);
   return (FALSE);
  }

  if (spi -> psp -> itm [i].pro.stt == STT_QUERY)
  {
   if (!sch_sts_resolve_t (spi,i,&spi -> psp -> itm [i].epi.tgt_m))
   {
    cci_error ("unresolved target '%s' in test '%s'",spi -> psp -> itm [i].epi.tgt_m.sym,spi -> psp -> itm [i].sym);
    return (FALSE);
   }
  }
 }

 return (TRUE);
}

/******************************************************************************/

%}

/******************************************************************************/
/* Token                                                                      */
/******************************************************************************/

/**************************************/
/* standard                           */
/**************************************/

%token TOK_INT	                        /* integer (obsolete) */
%token TOK_REAL	                        /* real               */
%token TOK_CHR	                        /* character          */
%token TOK_STR	                        /* string             */
%token TOK_BOOL                         /* boolean            */
%token TOK_ID	                        /* identifier         */
%token TOK_COMMENT                      /* comment            */
%token TOK_ERROR 	                    /* error              */
%token TOK_ROP                          /* relop              */
%token TOK_ACTION                       /* action             */

/**************************************/
/* Keywords                           */
/**************************************/

/******************/
/* Main           */
/******************/

%token KWD_STRUCTURE
%token KWD_MATCHING

/******************/
/* Field          */
/******************/

%token KWD_FIELD

%token KWD_TYPE
%token KWD_LABEL
%token KWD_START
%token KWD_LENGTH
%token KWD_MIN
%token KWD_MAX
%token KWD_TSP
%token KWD_EQU
%token KWD_ENC

/******************/
/* Result         */
/******************/

%token KWD_RESULT
%token KWD_PID
%token KWD_UPDATE
%token KWD_NOTIFICATION

%token KWD_NIL
%token KWD_GEN
%token KWD_GET

%token KWD_MESSAGE

/******************/
/* Status         */
/******************/

%token KWD_TEST
%token KWD_PROLOGUE
%token KWD_EPILOGUE
%token KWD_SURE
%token KWD_EXACT
%token KWD_KEY
%token KWD_OPTIONAL

%token KWD_INPUT

/******************/
/* symbol         */
/******************/

%token SYM_INT                          /* integer            */
%token SYM_FUNCTION                     /* function           */
%token SYM_VARIABLE                     /* variable           */
%token SYM_RULE                         /* rule               */
%token SYM_SYMBOL                       /* symbol             */

/******************/
/* special        */
/******************/

%token TOK_ARROW                        /* arrow              */

/******************************************************************************/
/*                                                                            */
/******************************************************************************/

%left '+' '-'
%left '^'
%left '|'
%left '&'
%left '<' '>'
%left '*' '/' '%'
%left UNARYMINUS '~'
%right POWER

%left KWD_AND KWD_OR
%right KWD_NOT

%start input

%%

/******************************************************************************/
/* Input                                                                      */
/******************************************************************************/

/**************************************/
/* Input                              */
/**************************************/

input               : fsp rsp psp
                    | TOK_ID
                      {
                       cci_error ("syntax error at '%s'",$1.tok.txt);
                      }
                    ;

/**************************************/
/* Construct                          */
/**************************************/
/*
spc                 : sch_fsp sch_rsp sch_psp
                      {
                       if (!sch_sts_resolve (spi))
                       {
                        cci_error ("inconsistent schema specification");
                       }
                      }

                    | TOK_ID
                      {
                       cci_error ("undefined symbol (%s)",$1.tok.txt);
                      }
                    ;
*/
/******************************************************************************/
/* Field Specification                                                        */
/******************************************************************************/

fsp                 : /* empty */
                    | fsp_lst
                    ;

/**************************************/

fsp_lst             : fsp_itm fsp_lst
                    | fsp_itm
                    ;

/**************************************/

fsp_itm             : KWD_FIELD TOK_ID '{' fld_lst '}'
                      {
                       $$ = $4;
                       strcpy ($$.fld.sym,$2.tok.txt);
                       if (spi -> fsp)
                        fsp_add (spi -> fsp,&$$.fld);
                       else
                        cci_error ("format specification not allowed here");
                      }
                    ;

/**************************************/

fld_lst             : fld_itm fld_lst
                      {
                       memcpy (&$$.fld,&$2.fld,sizeof ($$.fld));
                       sch_fld_pass (&$$.fld,&$1);
                      }
                    | fld_itm
                      {
                       memset (&$$.fld,0,sizeof ($$.fld));
                       sch_fld_pass (&$$.fld,&$1);
                      }
                    ;

/**************************************/

fld_itm             : KWD_TYPE '=' TOK_ID
                      {
                       DTP t;

                       if (!fsp_dtp_set	(&t,$3.tok.txt))
                        cci_error ("invalid data type");

                       $$.tok.idx = FSP_ITM_DTP;
                       $$.tok.num = t;
                      }

                    | KWD_LABEL '=' TOK_STR
                      {
                       $$.tok.idx = FSP_ITM_LBL;
                       strcpy ($$.tok.txt,$3.tok.txt);
                      }

                    | KWD_START '=' TOK_INT
                      {
                       $$.tok.idx = FSP_ITM_POS;
                       $$.tok.num = $3.tok.itg;
                      }

                    | KWD_LENGTH '=' TOK_INT
                      {
                       $$.tok.idx = FSP_ITM_LEN;
                       $$.tok.num = $3.tok.itg;
                      }

                    | KWD_MIN '=' TOK_INT
                      {
                       $$.tok.idx = FSP_ITM_MIN;
                       $$.tok.num = $3.tok.itg;
                      }

                    | KWD_MAX '=' TOK_INT
                      {
                       $$.tok.idx = FSP_ITM_MAX;
                       $$.tok.num = $3.tok.itg;
                      }

                    | KWD_TSP '=' TOK_STR
                      {
                       TFM t;

                       if (!fsp_tfm_dec	(&t,$3.tok.txt))
                        cci_error ("invalid transformation code");

                       $$.tok.idx = FSP_ITM_TFM;
                       $$.tok.num = t;
                      }

                    | KWD_EQU '=' TOK_STR
                      {
                       EQU equ;

                       fsp_equ_dec (&equ,$3.tok.txt);
                       $$.tok.idx = FSP_ITM_EQU;
                       $$.tok.num = equ;
                      }

                    | KWD_ENC '=' TOK_ID
                      {
                       CTT ctt;

                       if (!fsp_ctt_dec	(&ctt,$3.tok.txt))
                        cci_error ("invalid encryption type");

                       $$.tok.idx = FSP_ITM_CTT;
                       $$.tok.num = ctt;
                      }

                    | KWD_ENC '=' '-'
                      {
                       $$.tok.idx = FSP_ITM_CTT;
                       $$.tok.num = CTT_ZERO;
                      }


                    ;

/******************************************************************************/
/* Matching Specification                                                     */
/******************************************************************************/

/* sch_mat             : KWD_MATCHING sch_rsl sch_tst */

/**************************************/

rsp                 : /* empty */
                    | rsp_lst
                    ;

/**************************************/

rsp_lst             : rsp_itm rsp_lst
                    | rsp_itm
                    ;

/**************************************/

rsp_itm             : KWD_RESULT TOK_ID  '{' rsl_lst '}'
                      {
                       $$ = $4;
                       strcpy ($$.rsl.sym,$2.tok.txt);
                       if (spi -> rsp)
                        rsp_add (spi -> rsp,&$$.rsl);
                       else
                        cci_error ("result specification not allowed here");
                      }
                    ;

/**************************************/

rsl_lst             : rsl_itm rsl_lst
                      {
                       memcpy (&$$.rsl,&$2.rsl,sizeof ($$.rsl));
                       sch_rsl_pass (&$$.rsl,&$1);
                      }
                    | rsl_itm
                      {
                       memset (&$$.rsl,0,sizeof ($$.rsl));
                       sch_rsl_pass (&$$.rsl,&$1);
                      }
                    ;

/**************************************/

rsl_itm             : KWD_PID '=' TOK_INT
                      {
                       $$.tok.idx = RSP_ITM_PRM;
                       switch ($3.tok.itg)
                       {
                        case 0:		$$.tok.itg = PRM_ZERO;				break;
                        case 1:		$$.tok.itg = PRM_GET;				break;
                        default:	cci_error ("invalid pid retrieval mode");	break;
                       }

                      }

                    | KWD_PID '=' '*'
                      {
                       $$.tok.idx = RSP_ITM_PRM;
                       $$.tok.itg = PRM_GEN;
                      }

                    | KWD_UPDATE '=' TOK_INT
                      {
                       $$.tok.idx = RSP_ITM_RUM;

                       switch ($3.tok.itg)
                       {
                        case 0:		$$.tok.itg = RUM_ZERO;				break;
                        case 1:		$$.tok.itg = RUM_UPDATE;			break;
                        default:	cci_error ("invalid record update mode");	break;
                       }
                      }

                    | KWD_MESSAGE '=' stl
                      {
                       $$.tok.idx = RSP_ITM_MSG;
                       strcpy ($$.tok.txt,$3.tok.txt);
                      }

                    | KWD_NOTIFICATION '=' stl
                      {
                       $$.tok.idx = RSP_ITM_NOT;
                       strcpy ($$.tok.txt,$3.tok.txt);
                      }

                    ;


/******************************************************************************/
/* Matching Specification                                                     */
/******************************************************************************/

psp                 : /* empty */
                    | psp_lst
                      {
                       if (!sch_sts_resolve (spi))
                       {
                        cci_error ("inconsistent schema specification");
                       }
                      }
                    ;

/**************************************/

psp_lst             : psp_itm psp_lst
                    | psp_itm
                    ;

/**************************************/

psp_itm             : KWD_TEST TOK_ID  '{' tst_pro tst_epi '}'
                      {
                       memset (&$$.sts,0,sizeof ($$.sts));
                       memcpy (&$$.sts.pro,&$4.sts.pro,sizeof ($$.sts.pro));
                       memcpy (&$$.sts.epi,&$5.sts.epi,sizeof ($$.sts.epi));
                       strcpy ($$.sts.sym,$2.tok.txt);
                       if (spi -> psp)
                        psp_add (spi -> psp,&$$.sts);
                       else
                        cci_error ("procedure specification not allowed here");
                      }

                    | KWD_TEST TOK_INT '{' tst_pro tst_epi '}'
                      {
                       memset (&$$.sts,0,sizeof ($$.sts));
                       memcpy (&$$.sts.pro,&$4.sts.pro,sizeof ($$.sts.pro));
                       memcpy (&$$.sts.epi,&$5.sts.epi,sizeof ($$.sts.epi));
                       sprintf ($$.sts.sym,"%d",$2.tok.itg);
                       if (spi -> psp)
                        psp_add (spi -> psp,&$$.sts);
                       else
                        cci_error ("procedure specification not allowed here");
                      }
                    ;

tst_pro             : KWD_PROLOGUE '{' tst_pro_lst '}'
                      {
                       $$ = $3;
                       $$.sts.pro.stt = STT_QUERY;
                       $$.sts.pro.mtm = $3.sts.pro.mtm;
                      }

                    | KWD_PROLOGUE '{' KWD_INPUT KWD_SURE '?' '}'
                      {
                       memset (&$$.sts,0,sizeof ($$.sts));
                       $$.sts.pro.stt = STT_INPUT;
                       $$.sts.pro.mtm = 0x00;
                      }
                    ;

tst_epi             : KWD_EPILOGUE '{' tst_epi_lst '}'
                      {
                       $$ = $3;
                      }
                    ;

tst_pro_lst         : tst_pro_itm tst_pro_lst
                      {
                       memcpy (&$$.sts,&$2.sts,sizeof ($$.sts));
                       sch_sts_pass (&$$.sts,&$1);
                      }
                    | tst_pro_itm
                      {
                       memset (&$$.sts,0,sizeof ($$.sts));
                       sch_sts_pass (&$$.sts,&$1);
                      }
                    ;

tst_epi_lst         : tst_epi_itm tst_epi_lst
                      {
                       memcpy (&$$.sts,&$2.sts,sizeof ($$.sts));
                       sch_sts_pass (&$$.sts,&$1);
                      }
                    | tst_epi_itm
                      {
                       memset (&$$.sts,0,sizeof ($$.sts));
                       sch_sts_pass (&$$.sts,&$1);
                      }
                    ;

tst_pro_itm         : KWD_SURE '=' TOK_INT
                      {
                       $$.tok.idx = PSP_ITM_MTM;

                       switch ($3.tok.itg)
                       {
                        case 0:		$$.tok.itg = MTM_S0;				break;
                        case 1:		$$.tok.itg = MTM_S1;				break;
                        default:	cci_error ("invalid sureness specification");	break;
                       }
                      }

                    | KWD_SURE '=' '*'
                      {
                       $$.tok.idx = PSP_ITM_MTM;
                       $$.tok.itg = MTM_SX;
                      }

                    | KWD_EXACT '=' TOK_INT
                      {
                       $$.tok.idx = PSP_ITM_MTM;
                       switch ($3.tok.itg)
                       {
                        case 0:		$$.tok.itg = MTM_X0;				break;
                        case 1:		$$.tok.itg = MTM_X1;				break;
                        default:	cci_error ("invalid exactness specification");	break;
                       }
                      }

                    | KWD_EXACT '=' '-'
                      {
                       $$.tok.idx = PSP_ITM_MTM;
                       $$.tok.itg = MTM_XI;
                      }

                    | KWD_KEY '=' TOK_INT
                      {
                       $$.tok.idx = PSP_ITM_MTM;
                       switch ($3.tok.itg)
                       {
                        case 0:		$$.tok.itg = MTM_K0;				break;
                        case 1:		$$.tok.itg = MTM_K1;
                        break;
                        default:	cci_error ("invalid key specification");	break;
                       }
                      }

                    | KWD_OPTIONAL '=' TOK_INT
                      {
                       $$.tok.idx = PSP_ITM_MTM;
                       switch ($3.tok.itg)
                       {
                        case 0:		$$.tok.itg = MTM_O0;				break;
                        case 1:		$$.tok.itg = MTM_O1;				break;
                        default:	cci_error ("invalid optionality specification");	break;
                       }
                      }
                    ;

tst_epi_itm         : TOK_INT ':' TOK_ID
                      {
                       switch ($1.tok.itg)
                       {
                        case 0:
                         $$.tok.idx = PSP_ITM_TGT_0;
                         strcpy ($$.tok.txt,$3.tok.txt);
                         break;

                        case 1:
                         $$.tok.idx = PSP_ITM_TGT_1;
                         strcpy ($$.tok.txt,$3.tok.txt);
                         break;

                        default:
                         cci_error ("invalid epilogue case");
                         break;
                       }
                      }

                    | TOK_INT ':' TOK_INT
                      {
                       switch ($1.tok.itg)
                       {
                        case 0:
                         $$.tok.idx = PSP_ITM_TGT_0;
                         sprintf ($$.tok.txt,"%d",$3.tok.itg);
                         break;

                        case 1:
                         $$.tok.idx = PSP_ITM_TGT_1;
                         sprintf ($$.tok.txt,"%d",$3.tok.itg);
                         break;

                        default:
                         cci_error ("invalid epilogue case");
                         break;
                       }
                      }

                    | '*'     ':' TOK_ID
                      {
                       $$.tok.idx = PSP_ITM_TGT_M;
                       strcpy ($$.tok.txt,$3.tok.txt);
                      }

                    | '*'     ':' TOK_INT
                      {
                       $$.tok.idx = PSP_ITM_TGT_M;
                       sprintf ($$.tok.txt,"%d",$3.tok.itg);
                      }
                    ;


/******************************************************************************/
/* Support                                                                    */
/******************************************************************************/

stl                 : TOK_STR stl
                      {
                       $$.tok.idx = TOK_STR;
                       strcpy ($$.tok.txt,$1.tok.txt);
                       strcat ($$.tok.txt,$2.tok.txt);
                      }
                    | TOK_STR
                      {
                       $$.tok.idx = TOK_STR;
                       strcpy ($$.tok.txt,$1.tok.txt);
                      }
                    ;


%%

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/

static void psr_error (char * fmt,...)
{
 static char s [300];
 long l;
 va_list arg;

 va_start (arg,fmt);
 vsprintf (s,fmt,arg);
 va_end (arg);
 //l = vli_get_line ();

 printf (" ERROR [%04lu]: %s\n",l,s);

 // fprintf (stdout,"scn.dic = %P\n",scn.dic); getch ();
}

/*extern */ int yydebug;

CCI_YACC spc_yacc =
{
 &yydebug,
 (CCI_TOKEN *)&yylval,
 yyparse
};

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/




