/******************************************************************************/
/* psx-spi.c                                 Specification Parser Interface   */
/******************************************************************************/
/** @file psx-spi.c Specification Parser Interface - Source Code File
 * Definitions and functions supporting the usage of a generated parser for
 * parsing specification files.
 */

#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

extern CCI_LEX  spc_lex;
extern CCI_YACC spc_yacc;

static CCI_CFG spc_cfg =
{
 &spc_lex,
 &spc_yacc,
 ""
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** clear specification parser interface structure
 * @param spi specification parser interface structure
 */

static BOOL spi_clear (PSX_SPI * spi)
{
 if (spi == NULL)
  return (FALSE);

 if (spi -> fsp)
  fsp_delete (&spi -> fsp);

 if (spi -> rsp)
  rsp_delete (&spi -> rsp);

 if (spi -> psp)
  psp_delete (&spi -> psp);

 assert (spi -> fsp == NULL);
 assert (spi -> rsp == NULL);
 assert (spi -> psp == NULL);

 return (TRUE);
}

/******************************************************************************/
/** initialize spi, if scp AND SCP_RSP or PSP is true, create new rsp or psp
 * @param spi specification parser interface structure
 * @param scp specification component
 */

static BOOL spi_init (PSX_SPI * spi,SCP scp)
{
 if (spi == NULL)
  return (FALSE);

 memset (spi,0,sizeof (*spi));

 if (TEST (scp,SCP_FSP))
  if (!fsp_create (&spi -> fsp))
  {
   spi_clear (spi);
   return (FALSE);
  }

 if (TEST (scp,SCP_RSP))
  if (!rsp_create (&spi -> rsp))
  {
   spi_clear (spi);
   return (FALSE);
  }

 if (TEST (scp,SCP_PSP))
  if (!psp_create (&spi -> psp))
  {
   spi_clear (spi);
   return (FALSE);
  }

 return (TRUE);
}

/******************************************************************************/
/** check specification parser interface structure
 * @param spi specification parser interface structure
 * @param scp specification component
 */
 
static BOOL spi_check (PSX_SPI * spi,SCP scp)
{
 if (spi == NULL)
  return (FALSE);

 if (spi -> fsp -> n > 0 && !TEST (scp,SCP_FSP))   // format specification
  return (FALSE);

 if (spi -> rsp -> n > 0 && !TEST (scp,SCP_RSP))   // result specification
  return (FALSE);

 if (spi -> psp -> n > 0 && !TEST (scp,SCP_PSP))   // procedure specification
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/

extern int yydebug;
extern BOOL spi_psr_cfg (PSX_SPI * spi);

/** parse specification
 * @param spi specification parser interface
 * @param scp specification component [SCP_FSP|SCP_RSP|SCP_PSP]
 * @param f input format file name
 */
 
BOOL spi_parse (PSX_SPI * spi,SCP scp,PSX_STR f)
{
 int t;

 if (spi == NULL || f == NULL)
  return (FALSE);

 if (!spi_init (spi,scp))           // initialize spi and scp
  return (FALSE);

 strcpy (spc_cfg.input,f);

 if (!cci_cfg (&spc_cfg))
 {
  spi_clear (spi);
  return (FALSE);
 }

 if (!cci_init ())
 {
  spi_clear (spi);
  return (FALSE);
 }

 spi_psr_cfg (spi);

 yydebug = 0;

 t = spc_cfg.psr -> yyparse ();   // parse input file, set items of spi.fsp

 if (!cci_exit ())
 {
  spi_clear (spi);
  return (FALSE);
 }

 if (cci.error)
 {
  spi_clear (spi);
  return (FALSE);
 }

// if (psx.cfg.verbose)
//  sch_print (sch);

 return (TRUE);
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


