/******************************************************************************/
/* psx-stk.c                                      Stack Management Interface  */
/******************************************************************************/
/** @file psx-stk.c Stack Management Interface - Source Code File
 * Functions providing the functionalities of a stack datastructure.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** validate stack and set new number of elements
 * @param s stack structure
 * @param n new number of elements on stack
 */
static BOOL stk_validate (PSX_STK * s,NUM n)
{
 PTR * p;
 IDX i;

 if (s == NULL)                                   // check if stack exists
  return (FALSE);

 if ((s->s >= n) && (s->s <= n + 10))            // s gt n but lt n+10
 {                                       
  s -> n = n;                                    // new s->n is n
  return (TRUE);
 }

 if ((p = realloc (s -> p,n * s -> esz)) == NULL)  
  return (FALSE);                     // number of elements * element size
  
 for (i = s -> n;i < n;i++)
  //p [i] = 0;
  memset (PTR_OFS (p,i * s -> esz),0,s -> esz);   // initialize with 0

 s -> p = p;                                      // set stackpointer
 s -> s = n;                                      // last element ?
 s -> n = n;                                      // set number of element

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create and initialize stack
 * @param stk pointer to stack structure
 */
                                        
BOOL stk_create (PSX_STK ** stk)
{
 PSX_STK * t;

 if (stk == NULL)
  return (FALSE);

 if ((t = SYS_ALLOCATE (PSX_STK)) == NULL)    // get pointer to allocated memory
  return (FALSE);

 t -> p = NULL;                               // initialize with 0
 t -> n = 0;
 t -> s = 0;
 t -> esz = sizeof (PTR);

 *stk = t;                                    // link to stk

 return (TRUE);
}

/******************************************************************************/
/* delete Stack and free memory
 * @param stk pointer to stack structure
 */

BOOL stk_delete (PSX_STK ** stk)
{
 if (stk == NULL)
  return (FALSE);

 if (*stk == NULL)
  return (FALSE);

 if ((*stk) -> p)
  free ((*stk) -> p);

 free (*stk);
 *stk = NULL;

 return (TRUE);
}

/******************************************************************************/
/** create new stack element and copy element from pointer ptr into new element
 * @param stk pointer to stack structure
 * @param ptr pointer to new element
 */

BOOL stk_push (PSX_STK * stk,PTR ptr)
{
 IDX i;

 if (stk == NULL)
  return (FALSE);

 i = stk -> n;

 if (!stk_validate (stk,stk -> n + 1))    // new number of elements on stack
  return (FALSE);

 //stk -> p [i] = ptr;
 memcpy (PTR_OFS (stk -> p,i * stk -> esz),ptr,stk -> esz);
                             // copy element to new stack element
                             //sys: #define	PTR_OFS(p,o) (((char *) p) + (o))
 return (TRUE);
}

/******************************************************************************/
/** get/remove element from Stack
 * @param stk pointer to stack structure
 * @param ptr pointer to popped element
 */

BOOL stk_pop (PSX_STK * stk,PTR ptr)
{
 IDX i;

 if (stk == NULL)
  return (FALSE);

 if (stk -> n == 0)
  return (FALSE);

 i = stk -> n - 1;

 if (ptr)
  //*ptr = stk -> p [stk -> n - 1];
  memcpy (ptr,PTR_OFS (stk -> p,i * stk -> esz),stk -> esz);
                                                    // copy stack element to ptr

 if (!stk_validate (stk,stk -> n - 1))              // set new size of stack
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** set size of stack element
 * @param stk pointer to stack structure
 * @param esz new stack element size
 */

BOOL stk_set_esz (PSX_STK * stk,int esz)
{
 if (stk == NULL)
  return (FALSE);

 stk -> esz = esz;

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


