/******************************************************************************/
/* psx-stm.c                                            Stream Interface      */
/******************************************************************************/
/** @file psx-stm.c Stream Interface - Source Code File
 * Functions providing a stream management system.
 */
 
#include "psx.h"
#include "psx-stm.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get source stream
 * "f:test.txt" -> file, test.txt
 * "b"
 * @param src source string with information about stream
 * @param t type of stream (file or block)
 * @param ctl control, second character of prefix
 * @param sym name of stream (filename)
 */

static BOOL stm_src (char * src,TYPE * t,char * ctl,char * sym)
{
 char * p,pfx [SYS_STR_MAX];

 if (src == NULL || t == NULL)
  return (FALSE);

 if (strlen (src) > SYS_STR_MAX)
  return (FALSE);

 if ((p = strchr (src,':')) != NULL)      // get pointer to first ':'
 {
  int k = p - src;

  strncpy (pfx,src,k);                   // get prefix
  pfx [k] = 0;
  strcpy (sym,p + 1);                    // store stream name in sym
 }
 else
 {
  strcpy (pfx,src);                      // no prefix
  strcpy (sym,"");
 }

 if (strlen (pfx) == 0)                  
  return (FALSE);

 strcpy (ctl,pfx + 1);                   // control

 switch (pfx [0])
 {
  case 'f':                              // file
   *t = STP_FILE;
   break;

  case 'b':                              // block
   *t = STP_BLOCK;
   break;

  default:                               // nothing known
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** create stream: allocate structure memory, intitialize
 * @param stm pointer to stream information structure
 */

BOOL stm_create (SSI_STM ** stm)
{
 SSI_STM * s;

 if (stm == NULL)
  return (FALSE);

 if ((s = SYS_ALLOCATE (SSI_STM)) == NULL)
  return (FALSE);

 s -> ctx = NULL;


 *stm = s;

 return (TRUE);
}

/******************************************************************************/
/** delete stream: free structure space, free data space
 * @param stm pointer to stream structure
 */

BOOL stm_delete (SSI_STM ** stm)
{
 if (stm == NULL)
  return (FALSE);

 if (*stm == NULL)
  return (FALSE);

// if ((*stm) -> ptr)
//  free ((*str) -> ptr);

 free (*stm);
 *stm = NULL;

 return (TRUE);
}

/******************************************************************************/
/* open stream, create Stream (File oder Block)
 * @param stm pointer to stream structure
 * @param src source string
 */
/*
BOOL stm_open (SSI_STM ** stm,char * src)
{
 char ctl [SYS_STR_MAX],sym [SYS_STR_MAX];
 TYPE t;

 if (stm == NULL || src == NULL)
  return (FALSE);

 if (!stm_src (src,&t,ctl,sym))             // identify source type
  return (FALSE);

 if (!stm_create (stm))                     // create new stream
  return (FALSE);

 if (!stm_link (*stm,t,NULL))               // link stream
 {
  stm_delete (stm);
  return (FALSE);
 }

 if (!stm_ctl (*stm,STM_CMD_CREATE,0,&(*stm) -> ctx))
 {
  stm_delete (stm);
  return (FALSE);
 }

 if (!stm_ctl (*stm,STM_CMD_OPEN,0,sym))
 {
  stm_ctl (*stm,STM_CMD_DELETE,0,&(*stm) -> ctx);
  stm_delete (stm);
  return (FALSE);
 }

 return (TRUE);
}
*/
/******************************************************************************/
/* close stream, if existent
 */
/*
BOOL stm_close (SSI_STM ** stm)
{
 BOOL r = TRUE;

 if (stm == NULL)
  return (FALSE);

 if (*stm == NULL)
  return (FALSE);

 if (!stm_ctl (*stm,STM_CMD_CLOSE,0,0))
  r = FALSE;

 if (!stm_ctl (*stm,STM_CMD_DELETE,0,&(*stm) -> ctx))
  r = FALSE;

 if (!stm_delete (stm))
  r = FALSE;

 return (r);
}
*/
/******************************************************************************/
/** link stream (Block or File), initialize structure
 * @param stm pointer to stream structure
 * @param stp stream type
 * @param ctx pointer to context
 */

BOOL stm_link (SSI_STM * stm,TYPE stp,PTR ctx)
{
 if (stm == NULL)
  return (FALSE);

 switch (stp)
 {
  case STP_ZERO:
  {
   stm -> stp = STP_ZERO;
   stm -> ctl = NULL;
   stm -> ctx = NULL;
   break;
  }

  case STP_FILE:
  {
   stm -> stp = STP_FILE;
   stm -> ctl = stm_ctl_f;
   stm -> ctx = ctx;
   break;
  }

  case STP_BLOCK:
  {
   stm -> stp = STP_BLOCK;
   stm -> ctl = stm_ctl_b;
   stm -> ctx = ctx;
   break;
  }

  default:
   return (FALSE);
 }

 return (TRUE);
}

/******************************************************************************/
/**  do operation on stream
 * @param stm pionter to stream structure
 * @param cmd stream command (defined macro)
 * @param n index (information for control procedure call)
 * @param p pointer to string, also depending on command
 */
 
BOOL stm_ctl (SSI_STM * stm,IDX cmd,NUM n,PTR p)
{
 if (stm == NULL)
  return (FALSE);

 if (stm -> ctl == NULL)
  return (FALSE);

 if (!stm -> ctl (stm -> ctx,cmd,n,p))   // call control procedure: (stm_ctl_f() or stm_ctl_b())
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** read symbol from stream
 * @param stm pointer to sream structure
 * @param p pointer to string (return string)
 * @param n size of object to be read
 */

BOOL stm_read (SSI_STM * stm,PTR p,NUM n)
{
 return (stm_ctl (stm,STM_CMD_READ,n,p));
}

/******************************************************************************/
/** write symbol on stream
 * @param stm pointer to sream structure
 * @param p pointer to input string
 * @param size of object, that has to be written
 */

BOOL stm_write (SSI_STM * stm,PTR p,NUM n)
{
 return (stm_ctl (stm,STM_CMD_WRITE,n,p));
}

/******************************************************************************/
/** set index position in stream
 * @param stm pointer to stream structure
 * @param pos new position in stream
 */

BOOL stm_setpos (SSI_STM * stm,IDX pos)
{
 return (stm_ctl (stm,STM_CMD_SETPOS,pos,NULL));
}

/******************************************************************************/
/** get index position in stream
 * @param stm pointer to sream structure
 * @param pos retrieved position in stream
 */

BOOL stm_getpos (SSI_STM * stm,IDX * pos)
{
 return (stm_ctl (stm,STM_CMD_GETPOS,0,pos));
}

/******************************************************************************/
/** get ctx value
 * @param stm pointer to stream structure
 * @param stp stream type
 * @param pctx pointer to context
 */

BOOL stm_getctx (SSI_STM * stm,TYPE stp,PTR pctx)
{
 PTR * p;

 if (stm == NULL || pctx == NULL)
  return (FALSE);

 if (stm -> stp != stp)
  return (FALSE);

 if (stm -> ctx == NULL)
  return (FALSE);

 p = pctx;
 *p = stm -> ctx;

 return (TRUE);
}

/******************************************************************************/
/** read a line from stream
 * @param stm pointer to stream structure
 * @param p pointer to (output) string
 * @param n maximum number of retrieved characters (?)
 */

BOOL stm_readl (SSI_STM * stm,PTR p,NUM n)
{
 IDX  i;
 char c,* t;
 BOOL r;

 if (stm == NULL || p == NULL)
  return (FALSE);

 t = p;
 i = 0;
 r = FALSE;

 while (stm_read (stm,&c,1))
 {
  r = TRUE;

  if (c == 0x0A)                   // 0x0A = LF (line feed)
   break;

  if (i + 2 > n)
   return (FALSE);

  t [i++] = c;
 }

 if (!r)
  return (FALSE);

 t [i] = 0;

 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



