/******************************************************************************/
/* psx-str.c                                                String Functions  */
/******************************************************************************/
/** @file psx-str.c String Functions - Source Code File
 * Functions for string handling supporting the string processing in each module
 * by implementing some basic string operations often used in PSX:
 * - Converting a string to capital letters
 * - Checking for space characters
 * - NULL string notification
 * - DIN/ASCII conversion
 * - Accent conversion
 * - Formatting numbers
 * - Cutting off a specified character from end of string
 * - Cutting off blanks from front and back of string
 * and others
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** convert small characters into capital characters
 * @param src source string
 */
  
BOOL str_toupper (char * src)
{
 char *p1;
 int i;

 p1 = src;

 for (i = 0; i < strlen (src); i++, p1++)
 {
  if (*p1 >= 'a' && *p1 <= 'z')
  {
   *p1 = *p1 - ('a' - 'A');
  }
      else if (*p1 ==''){
        *p1 = '';
      }
      else if (*p1 ==''){
        *p1 = '';
      }
      else if (*p1 ==''){
        *p1 = '';
      }
      else
      {
        ;
      }
   }

 return (TRUE);
}

/******************************************************************************/
/** check each character of src string for space character
 * @param src source string
 */
  
BOOL str_isspace (const unsigned char * src)
{
 int i,n;

 if (src == NULL)
  return (FALSE);

 n = strlen (src);

 for (i = 0;i < n;i++)  
  
  //if (src[i] != ' ')
  if (!(isspace (src [i])))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check src string with str_isspace() for space character
 * @param src source string
 */

BOOL str_isnil (const char * src)
{
 if (src == NULL)
  return (TRUE);

 if (str_isspace (src))
  return (TRUE);

 return (FALSE);
}

/******************************************************************************/
/** check src string for character c, returns TRUE, if c is found
 * @param src source string
 * @param c character to be found in src
 */
  
BOOL str_contains (char * src,char * c)
{
 int i,n;

 if (src == NULL || c == NULL)
  return (FALSE);

 n = strlen (c);

 for (i = 0;i < n;i++)
  if (strchr (src,c [i]) != NULL)
   return (TRUE);

 return (FALSE);
}

/******************************************************************************/
/** get substring of src up to delimiter character del and store it in tok, store current position in idx
 * @param src input string
 * @param idx pointer to current position in src
 * @param del delimiter character
 * @param tok substring of src up to delimiter del
 */
 
BOOL str_tok (const char * src,int * idx,const char * del,char * tok)
{
 char * p,* t;
 int n,k,i,j;

 if (src == NULL || idx == NULL || del == NULL || tok == NULL)
  return (FALSE);

 n = strlen (src);

 if (*idx >= n)
  return (FALSE);

 p = src + *idx;  // p points to character at position idx
 k = n - *idx;

 i = 0;
 j = 0;

 while (i < k && (t = strchr (del,p [i])) == NULL)  // until delimiter found
  tok [j++] = p [i++];		// copy p to tok until the del character

 tok [j] = 0;  // terminate string
 *idx += i + (t ? 1 : 0);	// idx = current position, now index of del character

 return (TRUE);
}

/******************************************************************************/
/** get token from string, space is used as delimiter
 * @param src source string
 * @param idx strart index for token search
 * @param tok retrieved token
 */
  
BOOL str_tok_w (char * src,int * idx,char * tok)
{
 char * p;
 int n,k,i,j;
 BOOL q = FALSE;

 if (src == NULL || idx == NULL || tok == NULL)
  return (FALSE);

 n = strlen (src);

 if (*idx >= n)      // index must be less than length of src
  return (FALSE);

 p = src + *idx;     
 k = n - *idx;       // length of stringrest

 i = 0;
 j = 0;

 while (i < k && isspace (p [i]))    // count number of spaces
  i++;

 if (i >= k)
  return (FALSE);

 while (i < k && (!isspace (p [i]) || q))
 {
  if (p [i] == '"')                  
   q = !q;                           // BOOL q : quote, or nnot quote
  tok [j++] = p [i++];               // get next character
 }

 tok [j] = 0;                        // close string
 *idx += i;

 return (TRUE);
}

/******************************************************************************/
/** chop space character from string, result is one word in beginning of src
 * @param src source string
 */
BOOL str_trim (unsigned char * src)
{
 int i0,i1,n,k;

 if (src == NULL)
  return (FALSE);

 if ((n = strlen (src)) == 0)
  return (TRUE);

 i0 = 0;

 while (i0 < n && isspace (src [i0]))        // count space character
 //while (i0 < n && (src [i0] == ' ')) 
  i0++;

 i1 = n - 1;                                 // index of last character in string

 while (i1 >= 0 && isspace (src [i1]))       // get index of last non-space char
 //while (i1 >= 0 && (src [i1] == ' ')) 
  i1--;

 if (i0 > i1)                    // then string is empty
 {
  strcpy (src,"");
  return (TRUE);
 }

 k = i1 - i0 + 1;                // k : number of non-space characters

 if (k < 0)
 {
  strcpy (src,"");
  return (TRUE);
 }

 memmove (src,src + i0,k);      // move retrieved token to beginnig of string

 src [k] =  0;                  // close string
 return (TRUE);
}

/******************************************************************************/
/** append n chars to src, fill with blanks
 * @param src source string
 * @param n number of added space characters
 */

BOOL str_pad (char * src,int n)
{
 int i,k;

 if (src == NULL)
  return (FALSE);

 k = strlen (src);

 for (i = k;i < n;i++)
  strcat (src," ");

 return (TRUE);
}

/******************************************************************************/
/** chop brackets from source string
 * @param src source string
 * @param tgt target string
 * @param bl left bracket must be specified
 * @param br right bracket must be specified
 */
 
BOOL str_debrace (char * src,char * tgt,char bl,char br)
{
 PSX_STR t;
 int n;

 if (src == NULL || tgt == NULL)
  return (FALSE);

 strcpy (t,src);

 str_trim (t);
 n = strlen (t);

 if (n < 2)
  return (FALSE);

 if (!(t [0] == bl && t [n-1] == br))
  return (FALSE);

 strncpy (tgt,t + 1,n - 2);
 tgt [n - 2] = 0;

 return (TRUE);
}

/******************************************************************************/
/** generate string, initialize src with c
 * @param src source and result string
 * @param c initial character
 * @param n number of characters being filled with c 
 */

BOOL str_gen_c (char * src,char c,int n)
{
 if (src == NULL)
  return (FALSE);

 memset (src,c,n);    // set n characters of src to c
 src [n] = 0;         // close string

 return (TRUE);
}

/******************************************************************************/
/** remove character c from the end of string
 * @param src source string
 * @param c character that should be removed
 */
 
BOOL str_rem_chr_end (char * src,char c)
{
 int n,k;

 if (src == NULL)
  return (FALSE);

 if ((n = strlen (src)) == 0)
  return (FALSE);

 k = n - 1;

 if (src [k] == c)
  src [k] = 0;

 return (TRUE);
}

/******************************************************************************/
/** format number with leading zeros
 * @param src source string
 * @param len length of formatted number (e.g. 5 for 00123)
 */
 
BOOL str_fmt_num (char * src,int len)
{
 char t [1000],n [1000];
 int r,x;

 if (src == NULL)
  return (FALSE);

 r = atoi (src);            // convert string src to integer ( '123' -> 123 )

 sprintf (n,"%d",r);        // print r to n
 str_gen_c (t,'0',len);     // generate string t with number len '0'
 strcat (t,n);              // append n to t
 x = strlen (t) - len;      // x : index of n
 strcpy (src,t + x);        // copy n with leading '0's to src

 return (TRUE);
}


/******************************************************************************/
/** convert accented characters to german standard alphabet
 * @param s string pointer
 */
 
BOOL str_cvt_accent (char * str)
{
  char * p1, tmptxt[PSX_STR_MAX];
  int i;
  
  for (i = 0, p1 = str ; i < strlen (str) ; i++)
  {
    
      switch (*p1)
      {                        
        case '': case '': case '': case '': 
          tmptxt[i] = 'a';
          break;
        case '':
          tmptxt[i] = '';
          break;
        case '':
          tmptxt[i] = 'A';
          break;
        case '':
          tmptxt[i] = '';
          break;
        case '':
          tmptxt[i] = 'c';  
          break;
        case '':
          tmptxt[i] = 'C';
          break;
        case '': case '': case '': case '':
          tmptxt[i] = 'e';
          break;
        case '':
          tmptxt[i] = 'E';
          break;
        case '': case '': case '': case '':
          tmptxt[i] = 'i';
          break;
        case '':
          tmptxt[i] = 'n';
          break;
        case '':
          tmptxt[i] = 'N';            
          break;
        case '': case '': case '':
          tmptxt[i] = 'o';
          break;
        case '': case '': case '':
          tmptxt[i] = 'u';
          break;
        default:
          tmptxt[i] = *p1;
      } // end switch
    p1++;
    
  }
  
  tmptxt[i] = '\0';
  strncpy (str, tmptxt, strlen(tmptxt) );
  return (TRUE);
}

/******************************************************************************/
/** convert DIN 66003 encoded german specific letters into ISO 8859-1 format
 *  @param s string pointer, ISO 8850-1 converted after invocation
 */

BOOL str_cvt_din_grm (char * str) {

  char * p1, tmptxt[PSX_STR_MAX];
  int i;
 

  for (i = 0, p1 = str ; i < strlen (str) ; i++)
  {    
      switch (*p1)
      {
        case '[':
          tmptxt[i] = '';
          break;
        case '\\':
          tmptxt[i] = '';
          break;
        case ']':
          tmptxt[i] = '';
          break;
        case '{':
          tmptxt[i] = '';
          break;
        case '|':
          tmptxt[i] = '';
          break;
        case '}':
          tmptxt[i] = '';
          break;
        case '~':
          tmptxt[i] = '';
          break;
        default:
          tmptxt[i] = *p1;
      } // end switch
    
    p1++;

  }

  tmptxt[i] = '\0';
  strncpy (str, tmptxt, strlen(tmptxt) );
  return (TRUE);
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/




