/******************************************************************************/
/* psx-sys.c                                                 System Interface */
/******************************************************************************/
/** @file psx-sys.c System Interface - Source Code File
 * Functions providing the management of system functionalities, e.g. handling
 * of the commandline structure (argv is passed on to the Commandline Interface),
 * generation of messages, logging and memory management.
 */
 
#include "psx.h"
#include "psx-sys.inl"

#define SYS_DBG 0 

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

static struct
{
 SYS_ARG arg;
 char    apt [SYS_STR_MAX];
 BOOL    dbg;
} sys;

#ifdef malloc
#undef malloc
#endif

#ifdef realloc
#undef realloc
#endif

#ifdef free
#undef free
#endif

#ifdef strdup
#undef strdup
#endif


#include "psx-sys-dbg.inl"

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** initialize system
 * @param ac number of arguments
 * @param pointer to array of strings containing arguments
 */

BOOL sys_init (int ac,char ** av)
{
 sys.arg.c = ac;                        // number of paramaters of main call
 sys.arg.v = av;                        // {psx,command, {parameter}*, {option}*}
 sys.dbg   = FALSE;                     // debug mode

 sci_cfg (&sys.arg);                    // split parameter from main

 if (SYS_DBG)
  sys.dbg = TRUE;

 if (!dbg_init ())                      // initialize debug mode
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** exit system, exit debug mode
 */

BOOL sys_exit (void)
{
 if (!dbg_exit ())
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** get system arguments
 * @param arg pointer to system argument structure
 */

BOOL sys_arg_get (SYS_ARG * arg)
{
 if (arg == NULL)
  return (FALSE);

 *arg = sys.arg;

 return (TRUE);
}

/******************************************************************************/
/** get path from system argument structure (windows only)
 * @param pth pointer to string, path
 */

#ifndef WIN32
BOOL sys_apt_get (char * pth)
{
 char t [PATH_MAX],* p;

 if (pth == NULL)
  return (FALSE);

 if (realpath (sys.arg.v [0],t) == NULL)
	{
		  printf ("sys: realpath failed\n");
		  return (FALSE);
	}
 strcpy (pth,t); 

 return (TRUE);
}
#endif

/******************************************************************************/
/** output of argument list elements
 * @param fmt pointer to string containing format specification
 * @param l variable argument list
 */

BOOL sys_put_l (const char * fmt,va_list l)
{
 if (!cio_put_l (stdout,fmt,l))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** output using sys_put_l() with starting va_list
 * @param fmt pointer to constant format string
 * @param ... variable argument list
 */

BOOL sys_put (const char * fmt,...)
{
 BOOL r = TRUE;
 va_list l;


 va_start (l,fmt);
 r = sys_put_l (fmt,l);
 va_end (l);

 return (r);
}

/******************************************************************************/
/** generate message
 * @param fmt pointer to constant format string
 * @param l variable argument list
 */


BOOL sys_msg_l (const char * fmt,va_list l)
{
 const char * pfx = "> ";

 if (!sys_put ("%>%<",pfx))
  return (FALSE);

 if (!sys_put_l (fmt,l))       // output of list
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** start va_list and call sys_msg_l()
 * @param fmt pointer to constant format string
 * @param ... variable argument list
 */

BOOL sys_msg (const char * fmt,...)
{
 BOOL r = TRUE;
 va_list l;

 va_start (l,fmt);
 r = sys_msg_l (fmt,l);
 va_end (l);

 return (r);
}

/******************************************************************************/
/** write logfile
 * @param f file pointer to logfile
 * @param fmt pointer to constant format string
 * @param l variable argument list, contents for logfile
 */

BOOL sys_log_lf (FILE * f,const char * fmt,va_list l)
{
 const int pad_n = 22;
 SYS_STR t;

 if (f == NULL || fmt == NULL)
  return (FALSE);

 sys_log_time (t,pad_n);

 //if (!sys_put ("%>%<",t))
 // return (FALSE);

 if (!cio_put (f,"%>%<",t))
  return (FALSE);

 // if (!sys_put_l (fmt,l))
  // return (FALSE);

 if (!cio_put_l (f,fmt,l))                // write list to logfile
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** set outputstream for logging of list on stdout
 * @param fmt pointer to constant format string
 * @param l variable argument list
 */
 
BOOL sys_log_l (const char * fmt,va_list l)
{
 sys_log_lf (stdout,fmt,l );

 return (TRUE);
}

/******************************************************************************/
/** start variable argument list for logging, start logging of list
 * @param fmt pointer to constant format string
 * @param ... variable argument list
 */
 
BOOL sys_log (const char * fmt,...)
{

 BOOL r = TRUE;
 va_list l;

 va_start (l,fmt);           // get argument list
 r = sys_log_l (fmt,l);      // start logging
 va_end (l);

 return (r);                 // returns always TRUE
}

/******************************************************************************/
/** get environment for atr
 * @param atr pointer to constant string, attribute
 * @param val pointer to string, value of environment
 */

BOOL sys_env_get (const char * atr,char * val)
{
 char * t;

 if (atr == NULL || val == NULL)
  return (FALSE);

 strcpy (val,"");

 if ((t = getenv (atr)) == NULL)
  return (FALSE);

 strcpy (val,t);

 return (TRUE);

}


/******************************************************************************/
/** print failure message
 * @param fmt pointer to constant format string
 */
 
BOOL sys_fail (const char * fmt,...)
{
 char t [4000];

 if (fmt)
 {
  va_list l;

  va_start (l,fmt);
  vsprintf (t,fmt,l);
  va_end (l);
 }
 else
  strcpy (t,"???");

 printf ("\nSYS: fail: %s\n",t);
 exit (-1);

 return (TRUE);	// never
}

/******************************************************************************/
/* Public                                                   Memory Management */
/******************************************************************************/
/** allocate memory for pointer
 * @param p void pointer to new allocated memory
 * @param size of allocated memory
 */
 
BOOL sys_allocate (PTR p,NUM n)
{
 PTR * t,h;

 if (p == NULL)
  return (FALSE);

 t = (PTR *) p;

 if ((h = (PTR) malloc (n)) == NULL)

 {
  *t = NULL;
  return (FALSE);
 }

 *t = h;

 dbg_mem_allocate (*t);

 return (TRUE);
}

/******************************************************************************/
/** reallocate memory for an existing object
 * @param p pointer to memory
 * @param n new size of allocated memory
 */
 
BOOL sys_reallocate (PTR p,NUM n)
{
 PTR * t,h0,h1;

 if (p == NULL)
  return (FALSE);

 t = (PTR *) p;
 h0 = *t;

 if ((h1 = (PTR) realloc (*t,n)) == NULL)
 {
  return (FALSE);
 }

 *t = h1;

 dbg_mem_reallocate (h0,h1);

 return (TRUE);
}

/******************************************************************************/
/** free memory
 * @param p pointer to memory block
 */
 
BOOL sys_deallocate (PTR p)
{
 PTR * t,h;

 if (p == NULL)
  return (FALSE);


 t = (PTR *) p;

 if (*t == NULL)
  return (FALSE);

 h = *t;
 free (*t);
 *t = NULL;

 dbg_mem_free (h);

 return (TRUE);
}

/******************************************************************************/
/** allocate memory of specified size
 * @param n size of allocated memory
 * @return pointer to allocated memory 
 */
 
PTR sys_malloc (NUM n)
{
 PTR t;

 if (!sys_allocate (&t,n))
  return (NULL);

 return (t);
}

/******************************************************************************/
/** reallocate memory of specified size
 * @param p pointer to allocated memory
 * @param n size of requested memory
 * @return pointer to reallocated memory
 */
 
PTR sys_realloc (PTR p,NUM n)
{
 if (!sys_reallocate (&p,n))
  return (NULL);

 return (p);
}



/******************************************************************************/
/** free memory
 * @param p pointer to memory block
 */
 
void sys_free (PTR p)
{
 if (!sys_deallocate (&p))
  return;

 return;
}

/******************************************************************************/
/** allocate new memory and copy a string source
 * @param src pointer to source string
 * @return pointer to copy of src (physical copy of source string)
 */

char * sys_strdup (const char * src)

{

 char * t;
 int    n;

 if (src == NULL)
  return (NULL);

 n = strlen (src);

 if (!sys_allocate (&t,n + 1))
  return (NULL);

 memcpy (t,src,n);
 t [n] = 0;

 return (t);
}

/******************************************************************************/
/* Debug Management                                                           */
/******************************************************************************/
/** begin debug memory management
 */
 
BOOL sys_dbg_mem_begin ()
{

 SYS_MCX c;

 if (!dbg_mcx_get (&c))
  return (FALSE);

 if (!dbg_mcx_push (&c))
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** end debug memory management
 * @param file pointer to string, filename
 * @param line number of line
 */
 
BOOL sys_dbg_mem_end (const char * file,int line)
{
 SYS_MCX c0,c1;

 if (!dbg_mcx_pop (&c0))
  return (FALSE);

 if (!dbg_mcx_get (&c1))
  return (FALSE);

 if (c0.sem != c1.sem)
 {
  sys_fail ("[%s,%d]: memory semaphore: s0: %d, s1: %d",file,line,c0.sem,c1.sem);
  return (FALSE);	// never
 }

 return (TRUE);
}

/******************************************************************************/
/**  get debug management memory semaphore
 * @param sem pointer to value of semaphore
 */
 
BOOL sys_dbg_mem_get_sem (int * sem)
{
 if (sem == NULL)
  return (FALSE);

 *sem = sys_dbg.sem;


 return (TRUE);
}

/******************************************************************************/
/** execute system (non-windows only), start new process
 * @param c constant string
 * @param i character
 * @param o character, option for sys_execute_fork()
 * @param e pointer to character, environment for sys_execute_fork()
 * @param r return value of sys_execute_fork()
 */
 
#ifndef WIN32
BOOL sys_execute (const char * c,char * i,char ** o,char ** e,int * r)
{
 SYS_ARG a;

 if (c == NULL)
  return (FALSE);

 if (o)
  *o = NULL;

 if (e)
  *e = NULL;

 if (!arg_parse (&a,(char *) c))
  return (FALSE);

 *r = sys_execute_fork (a.v,i,o,e);

 arg_free (&a);

 return (TRUE);
}
#endif
/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


