/******************************************************************************/
/* psx-sys.inl                                               System Interface */
/******************************************************************************/
/** @file psx-sys.inl System Interface - Inline Source Code File
 * Functions supporting the functionality of the System Interface like argument
 * handling.
 */


//LINKAGE	BOOL	arg_set		(ARG * arg,int ac,char ** av);
//LINKAGE	BOOL	arg_get		(ARG * arg,char ** a);
//LINKAGE	BOOL	arg_parse	(ARG * arg,char * s);
//LINKAGE	BOOL	arg_free	(ARG * arg);
//LINKAGE	BOOL	arg_print	(ARG * arg);


/******************************************************************************/
/** add s to argument a
 * @param a target argument structure
 * @param s source string
 * @param n number of arguments
 */

static BOOL arg_add (ARG * a,char * s,NUM * n)
{
 char ** p,* t;
 int k;

 if (a == NULL || n == NULL)
  return (FALSE);

 if (s)
 {
  int i;

  if (a -> s == NULL)
   *n = 0;                         // no s available

  k = strlen (s) + 1;              // get length of s

  if ((t = (char *) realloc (a -> s,*n + k)) == NULL)    // extend a->s
   return (FALSE);

  for (i = 0;i < a -> c;i++)
   a -> v [i] = t + (a -> v [i] - a -> s);               // insert s in v[i]    
  memcpy (t + *n,s,k);	
  a -> s = t;                                            // store t in argument
  t = a -> s + *n;
  *n = *n + k;                    // n points to next free position (end of field)
 }
 else
  t = NULL;

 if ((p = (char **) realloc (a -> v,(a -> c + 1) * sizeof (char *))) == NULL)
  return (FALSE);

 p [a -> c] = t;
 a -> v = p;

 if (t)
  a -> c++;                       // increase counter if something was appended

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** set argument structure values
 * @param arg target argument structure
 * @param ac argc from main() function, number of arguments
 * @param av argv from main() function, argument list
 */

BOOL arg_set (ARG * arg,int ac,char ** av)
{
 if (arg == NULL)
  return (FALSE);

 arg -> c = ac;            // parameter argc from main()
 arg -> v = av;            // parameter argv from main()
 arg -> s = NULL;          // storage

 return (TRUE);
}

/******************************************************************************/
/** fetch argument value from arg
 * @param arg source argument structure
 * @param a argument value from argument list
 */

BOOL arg_get (ARG * arg,char ** a)
{
 if (arg == NULL)
  return (FALSE);

 if (arg -> c == 0)                  // no values v exist
  return (FALSE);

 arg -> c--;                         // decrement counter

 if (a)
  *a = *arg -> v;                    // get values

 arg -> v++;                         // go to next value

 return (TRUE);
}

/******************************************************************************/
/** add string s to argument structure arg
 * @param arg target argument structure
 * @param s source string
 */
 
BOOL arg_parse (ARG * arg,char * s)
{
 int i,i0;
 char t [SYS_STR_MAX];
 NUM n = 0;

 if (arg == NULL || s == NULL)
  return (FALSE);

 memset (arg,0,sizeof (*arg));

 //if ((arg -> s = (char *) malloc (strlen (s) + 1)) == NULL)

 i = 0;

 while (s [i] != 0)
 {
  int k;

  while ((s [i] == ' ') || (s [i] == '\t'))    // jump over leading whitespace 
   i++;

  i0 = i;                      // i0 = position of first character behind whitespace

  while ((s [i] != '\0') && (s [i] != ' ') && (s [i] != '\t'))
   i++;                        // i = position of last character in front of whitespace

  k = i - i0;                  // k = number of characters
  assert (k < SYS_STR_MAX);
  memcpy (t,s + i0,k);
  t [k] = 0;                   

  if (!arg_add (arg,t,&n))     // insert into arg
   return (FALSE);
 }

 if (!arg_add (arg,NULL,&n))   // close arg with NULL element
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** free allocated memory of arg
 * @param arg pointer to argument structure
 */

BOOL arg_free (ARG * arg)
{
 if (arg == NULL)
  return (FALSE);

 if (arg -> s == NULL)
  return (FALSE);

 free (arg -> s);

 if (arg -> v)
  free (arg -> v);

 memset (arg,0,sizeof (*arg));

 return (TRUE);
}

/******************************************************************************/
/** print arg on stdout
 * @param arg argument structure
 */

BOOL arg_print (ARG * arg)
{
 IDX i;

 if (arg == NULL)
  return (FALSE);

 printf ("arg [%p]:\n",arg);

 for (i = 0;i < arg -> c;i++)
  printf (" %d: '%s'\n",i,arg -> v [i]);

 return (TRUE);
}



/******************************************************************************/
/*                                                                            */
/******************************************************************************/
/** store current time in str
 * @param str target string
 */
 
static BOOL sys_get_str_time (char * str)	
{
 time_t      t;
 struct tm * tm;

 if (str == NULL)
  return (FALSE);

 t = time (0);
 tm = localtime (&t);
 assert (tm != NULL);
 // sprintf (str,"%d-%d-%d %d:%d:%d",1900 + tm -> tm_year,tm -> tm_mon + 1,tm -> tm_mday,t -> tm_hour,tm -> tm_min,tm -> tm_sec);
 strftime (str,SYS_STR_MAX,"%Y-%m-%d %H:%M:%S",tm);

 return (TRUE);
}

/******************************************************************************/
/** get current time and add blanks, close string with 0
 * @param str target string
 * @param pad number of characters in one line/column
 */

static BOOL sys_log_time (char * str,int pad)
{
 int l;

 if (str == NULL)
  return (FALSE);

 if (!sys_get_str_time (str))         // get time to str
  return (FALSE);

 if ((l = strlen (str)) < pad)
 {
  memset (str + l,' ',pad - l);       // fill with blanks
  str [pad] = 0;                      // terminate string with 0
 }

 return (TRUE);
}

/******************************************************************************/
/** add memory to str
 * @param str pointer to string
 * @param n new size of str
 */

static BOOL sys_str_realloc (char ** str,NUM n)
{
 PTR p;

 if (str == NULL)
  return (FALSE);

 if ((p = (PTR) realloc (*str,n)) == NULL)
  return (FALSE);

 *str = (char *) p;

 return (TRUE);
}

/******************************************************************************/
/** read from file descriptor f and store in s
 * @param f file descriptor
 * @param s stringbuffer
 */

static BOOL sys_str_read (int f,char ** s)
{
 char * t;
 char   c;
 NUM    n;

 t = NULL;
 n = 0;

 if ((t = (char *) malloc (1)) == NULL)
  return( FALSE);

 t [0] = 0;

 while (1)
 {
  int k = read (f,&c,1);

  switch (k)               // number of read bytes
  {
   case -1:                // read function call failed
   {
    if (t)
     free (t);

    return (FALSE);
   }

   case 0:                // no bytes read, eof
   {
    if (t == NULL)        // empty file
     return (FALSE);
     
    *s = t;
    return (TRUE);
   }

   default:
   {
    int i = n;

    assert (k == 1);
    n += k;                // n is total number of byte 

    if (!sys_str_realloc (&t,n + 1))
    {

     if (t)
      free (t);
     return (FALSE);
    }

    t [i] = c;             // store buffer 
    t [i + 1] = 0;
    break;
   }
  }
 }

 return (FALSE);
}

/******************************************************************************/
/** produce another process and execute a, communication through pipes
 * @param a pointer to pointer, name of executable file
 * @param i instruction to be sent by pipe ( father  -> child)
 * @param o option (?) to be sent by pipe (child -> father)
 * @param e environment (?) to be sent by pipe (child -> father)
 */
 
#ifndef WIN32

static int sys_execute_fork (char ** a,char * i,char ** o,char ** e)
{
 int pid,sts,r;
 int pi [2],po [2],pe [2];

 if (o)
  *o = NULL;

 if (e)
  *e = NULL;

 ///////////////////////////////////////

 if (pipe (pi) == -1)
  return (-1);

 if (pipe (po) == -1)
  return (-1);

 if (pipe (pe) == -1)
  return (-1);
 
 ///////////////////////////////////////

 if ((pid = fork ()) < 0)
 {
  perror ("fork");
  exit (1);
 }

 ///////////////////////////////////////

 if (pid == 0)          // child process
 {
  dup2 (pi [0],0); close (pi [1]);       // child listens only to instructions
  dup2 (po [1],1); close (po [0]);       
  dup2 (pe [1],2); close (pe [0]);

  execvp (*a,a);        // (filename, argv)
  perror (*a);
  exit (1);
 }

 close (pi [0]);                         // father gives instructions
 close (po [1]);                         // child sends o and e
 close (pe [1]);

 if (i)
  write (pi [1],i,strlen (i));

 close (pi [1]);

 while (wait (&sts) != pid)
 /* empty loop */;

 ///////////////////////////////////////

 if (o)
  if (!sys_str_read (po [0],o))
   r = FALSE;

 if (e)
  if (!sys_str_read (pe [0],e))
   r = FALSE;

 close (po [0]);
 close (pe [0]);

 if (WIFEXITED (sts))
  r = WEXITSTATUS (sts);
 else
  r = -1;

 return (r);
}
#endif

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


