/******************************************************************************/
/* psx-tui.c                                         Template User Interface  */
/******************************************************************************/
/** @file psx-tui.c Template User Interface - Source Code File
 * Functions and definitions for HTML document template management, if no
 * template specifications are available the automatic generation is invoked.
 */


#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

typedef BOOL (* TPL_PRC)(TUI_CTX * ctx);
#define TPL_PROCEDURE(prc) BOOL prc (TUI_CTX * ctx)

#include "psx-tui.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** Template index structure                                                  */
static const struct _TPL_IDX
{
 TPL     tpl;                  /**< template identification    */
 const char * sym;             /**< name of template           */
 TPL_PRC prc;                  /**< template procedure address */
} tpl_idx [] =
{
 { TPL_PTL,"tpl.ptl",tpr_ptl },        /* portal  */
 { TPL_SYS,"tpl.sys",tpr_sys },        /* system  */
 { TPL_REQ,"tpl.req",tpr_req },        /* request */
 { TPL_RET,"tpl.ret",tpr_ret },        /* return  */
 { TPL_MSG,"tpl.msg",tpr_msg },        /* message */
 { TPL_CFM,"tpl.cfm",tpr_cfm },        /* confirm */
 { TPL_NIL,"tpl.nil",tpr_nil }         /* error   */
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** lookup correct template.
 * Determines the correct document template for the current command.
 * A function to generate the appropriate default template is assigned
 * in prc, a config string indicating the template is assigned to cfg.
 * If prc or cfg is set to NULL, no function pointer or config string
 * will be returned.
 * @param tpl template identificator
 * @param prc pointer to template procedure
 * @param cfg string pointer to config string
 */

static BOOL tpl_lookup (TPL tpl,TPL_PRC * prc,char ** cfg)
{
 IDX i;
 NUM n;

 n = sizeof (tpl_idx) / sizeof (tpl_idx [0]);  /* get size of structure */

 for (i = 0;i < n;i++)
  if (tpl_idx [i].tpl == tpl)
  {
   if (prc)
    *prc = tpl_idx [i].prc;       /* get procedure     */

   if (cfg)
    *cfg = tpl_idx [i].sym;       /* get config string */
   return (TRUE);
  }

 return (FALSE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** load html document
 * @param ctx pointer to  TUI context structure
 * @param doc pointer to html document after invocation
 */
 
static BOOL tpl_doc_load (TUI_CTX * ctx,PSX_DOC * doc)
{
 char * t;
 PSX_STR f;

 if (ctx == NULL || doc == NULL)
  return (FALSE);

 if (!tpl_lookup (ctx -> tpl,NULL,&t))   /* lookup template         */
  return (FALSE);

 if (!psx_cfg_get (t,f))                 /* get config              */
  return (FALSE);

 if (str_isnil (f))
  return (FALSE);

 if (!doc_load (doc,f))                  /* load document from file */
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** invoke creation of html document
 * @param ctx pointer to TUI context
 * @param doc pointer to html document
 */
 
static BOOL tpl_doc_make (TUI_CTX * ctx,PSX_DOC * doc)
{
 TPL_PRC prc;
 BOOL r;

 if (ctx == NULL || doc == NULL)
  return (FALSE);

 if (!tpl_lookup (ctx -> tpl,&prc,NULL))
  return (FALSE);

 if (!hci_begin ())
  return (FALSE);

 r = TRUE;

 assert (prc != NULL);

 if (!prc (ctx))                       /* get html code  */
  r = FALSE;

 //hci_dump ();

 if (!hci_put_d (doc))                 /* print document */
  r = FALSE;

 if (!hci_end ())
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** get template document
 * @param ctx pointer to TUI context structure
 * @param doc pointer to html document
 */
 
static BOOL tpl_doc_get (TUI_CTX * ctx,PSX_DOC * doc)
{
 BOOL r = TRUE;

 if (ctx == NULL || doc == NULL)
  return (FALSE);

 if (!tpl_doc_load (ctx,doc))       /* load document from template, if existent */
  if (!tpl_doc_make (ctx,doc))      /* fill document with html code             */
   return (FALSE);

 if (!doc_compile (doc,ctx -> dic)) /* compile document                         */
  r = FALSE;

 return (r);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** put document to output
 * @param ctx pointer to TUI context structure
 * @param doc pointer to html document
 */
 
BOOL tui_doc_put (TUI_CTX * ctx,PSX_DOC * doc)
{
 if (ctx == NULL || doc == NULL)
  return (FALSE);

 if (doc -> txt == NULL)
  return (FALSE);

 cgi_put ("%s",doc -> txt);   /* cgi output */

 return (TRUE);
}

/******************************************************************************/
/** invoke html output
 * @param ctx pointer to TUI context structure
 */
 
BOOL tui_tpl_put (TUI_CTX * ctx)
{
 PSX_DOC * doc;
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (!doc_create (&doc))        /* create document   */
  return (FALSE);

 if (tpl_doc_get (ctx,doc))     /* fill document     */
 {
  if (!tui_doc_put (ctx,doc))   /* invoke cgi output */
   r = FALSE;
 }
 else
  r = FALSE;

 if (!doc_delete (&doc))
  r = FALSE;

 return (r);
}

/******************************************************************************/
/** invoke html document output
 * @param ctx pointer to TUI context structure
 */
 
BOOL tui_put (TUI_CTX * ctx)
{
 PSX_DOC * doc;
 BOOL r = TRUE;

 if (ctx == NULL)
  return (FALSE);

 if (!doc_create (&doc))        /* create document            */
  return (FALSE);

 if (tpl_doc_get (ctx,doc))     /* fill document              */
 {
  //printf ("%s",HCI_CNT);

  if (!tui_doc_put (ctx,doc))   /* invoke cgi output          */
   r = FALSE;
 }
 else
  r = FALSE;

 if (!doc_delete (&doc))        
  r = FALSE;

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


