/******************************************************************************/
/* psx-txt.c                               Text Operation Interface           */
/******************************************************************************/
/** @file psx-txt.c Text Operation Interface - Source Code File 
 * Functions and definitions providing text operations
 * - search in a string array for a member that matches source string
 * - check whether string is name affix
 * - decomposition of name into up to three fractions
 * - initialisation of a string with up a number of specified characters
 * - padding of a string with spaces up to given length
 * - removal of all spaces at end of string
 * - conversion of german umlauts
 * - reduction multiple separators between two syllables to maximal one separator
 * - check whether a specifierd character is a delimiter
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#define ZEILE       1000        /**< maximal length of line                   */ 
#define VARIABLE     100        /**< maximal length of variable               */ 
#define MAXVARIABLE   30        /**< maximal number of variables              */ 

#define TRENNZEICHEN " .:,;-\'" /**< separators: [.][:][,][;][-][']           */ 

#define NAMENSBESTANDTEILE \
        {"AL", "AM", "AN", "AUF", \
         "D","DA", "DE", "DEL", "DELA", "DEM", "DEN", "DER", "DI", "DOS", "DR", "DU", \
         "EL", "EN", "ET", \
         "FREIFRAU", "FREIHERR", \
         "GRAEFIN", "GRAF", \
         "LA", "LE", \
         /*"MAC",*/ "MC", "MED", \
         "O", \
         "PD", "PROF", \
         "SR", \
         "UND", \
         "V", "VAN", "VO", "VOM", "VON", \
         "Y", \
         "ZU", "ZUM", "ZUR" }   /**< name affixes */

//         "MED", "MC", \

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/** search string array for a member that matches source string 
 * @param s string pointer to source string
 * @param liste array of string pointers to member values
 * @param unten int pointer to lower array index 
 * @param oben int pointer to upper array index
 * @return matching member if found
 * @return NULL if no match found
 */
 
static const char *suche (char *s, const char *liste[], int *unten, int *oben)
{
 int mitte; 
 int ergebnis; 

 while (*unten <= *oben)
 {
  mitte = (*unten + *oben)/2; 
  ergebnis = strcmp(s, liste[mitte]); 

  if (ergebnis < 0)
  { 
   *oben = mitte - 1;
  }
  else
   if (ergebnis > 0)
   { 
    *unten = mitte + 1; 
   }
   else   // s matches name component
   { 
    return(liste[mitte]);
   }
 } 
	 
 return (NULL); 
} 

/******************************************************************************/
/** check whether string is name affix
 * @param src pointer to source string
 * @return TRUE if source string is name affix
 * @return FALSE if source string is not name affix
 *
 */

static BOOL txt_isNameComp (char * src)
{ 
 int  flag = FALSE; 
 const char *namensbestandteile[] = NAMENSBESTANDTEILE; 
 int anzahl = sizeof(namensbestandteile) / sizeof(namensbestandteile[0]); 
 char *p;
 int unten, oben; 
 
 unten = 0; 
 oben = anzahl-1;      // number of name components
 
 p = suche (src,namensbestandteile,&unten,&oben);  // p now contains name component or NULL
 
 if (p != NULL)
 { 
  flag = TRUE; 
 } 

 // printf ("isNameComp (%s):%d\n",src,flag);
 return (flag);
} 
 
/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

/** decompose name into up to three fractions
 * @param src pointer to source string (name)
 * @param s1 pointer to string containing first fraction after invocation
 * @param s2 pointer to string containing second fraction after invocation
 * @param s3 pointer to string containing third fraction after invocation
   (i.g. rest of components and name affixes) 
*/ 
 
static BOOL txt_dec_n (char * src,char * s1,char * s2,char * s3)
{ 
 char namensbestandteil[2*VARIABLE];  // VARIABLE = 100
 char *n = namensbestandteil;         // name affix
 char array[VARIABLE]; 
 char *p = array; 
 int i = 1; 
 
 strcpy (n,""); 
 strcpy (s1,"");
 strcpy (s2,"");
 strcpy (s3,"");
 
 p = strtok (src,TRENNZEICHEN);  // p now contains first component of name
                                 // if separator found

 while (p)                       // only if separator found
 {
  if (txt_isNameComp (p))        // check if p is name affix
  {
   if (strcmp (n,"")!=0)
   {
    strcat (n," ");
   } 
   strcat (n,p);
  } 
  else                           // p is not name affix but regular component
  {
   switch (i)
   {
    case 1:
     ++i; 
     strcpy (s1,p); 
     break;

    case 2:
     ++i;
     strcpy (s2,p); 
     break;

    case 3:
     ++i;
     strcpy (s3,p); 
     break;

    default:
     strcat(s3," ");
     strcat (s3,p);
     break;
   } 
  } 
  p = strtok(NULL, TRENNZEICHEN);   // get next name component
 }  // end while
 
 if (strcmp(n,"")!=0)               // add name affixes to third name component
 { 
  if (strcmp (s3,"")!=0)
  {
   strcat (s3," "); 
  } 
  strcat (s3,n); 
 } 

 return (TRUE);
} 


/******************************************************************************/
/** decompose first name into up to three fractions
 * @param src pointer to source string (name)
 * @param s1 pointer to string containing first fraction after invocation
 * @param s2 pointer to string containing second fraction after invocation
 * @param s3 pointer to string containing third fraction after invocation
*/ 

static BOOL txt_dec_f (char * src,char * s1,char * s2,char * s3)
{
 char array[2*VARIABLE];
 char *p = array;
 int i = 1;

 strcpy (s1,"");
 strcpy (s2,"");
 strcpy (s3,"");

 p = strtok (src,TRENNZEICHEN);

 while (p)
 {
  switch (i)
  {
   case 1:
    ++i; 
    strcpy (s1,p);
    break;

   case 2:
    ++i;
    strcpy (s2,p);
    break;

   case 3:
    ++i;
    strcpy (s3,p);
    break;

   default:
    strcat (s3," ");
    strcat (s3,p);
    break;
  }

  p = strtok (NULL,TRENNZEICHEN);
 }

 return (TRUE);
}
 
/******************************************************************************/

 
/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** initialize string with character c up to n characters
 * @param src pointer to source string
 * @param c character
 * @param n number of initialized characters
 */
 
BOOL txt_gen_chr (char * src,char c,int n)
{
 int i;

 if (src == NULL)
  return (FALSE);

 for (i = 0;i < n;i++)
  src [i] = c;

 src [n] = 0;

 return (TRUE);
}

/******************************************************************************/
/** pad string with spaces up to given length
 * @param txt PSX_STR to be padded, contains padded string after invocation
 * @param n length of txt after padding
 */
 
BOOL txt_val_len (PSX_STR txt,NUM n)
{
 int k;

 if (txt == NULL)
  return (FALSE);

 k = strlen (txt);

 while (k++ < n)
  strcat (txt," ");   // pad txt with spaces upto string length n

 txt [n] = 0;

 return (TRUE);
}

/******************************************************************************/
/** checks whether string has multiple components by searching for separators
 * @param s pointer to source string
 * @return TRUE if string contains separators, consist of multiple components
 * @return FALSE if string consist of one component only
*/ 

BOOL txt_isMultiPart (char * src)
{
 int  mehrteilig = FALSE;
 char *p1, *p2;
 char trennzeichen[]=TRENNZEICHEN;
 
 for (p1 = src; *p1 != '\0'; p1++)
 { 
  for (p2 = trennzeichen; *p2 != '\0' && *p2 != *p1; p2++) 
   ; 
  if (*p2 != '\0')
  { 
   mehrteilig = TRUE; 
   break; 
  } 
 } 

 return (mehrteilig); 
} 

/******************************************************************************/
/** remove all spaces end of string
 * @param src pointer to source string, contains trimmed string after invocation
 */
 
// korrigierte Version, behandelt auch Strings, die nur Leerzeichen ent- 
// halten, korrekt, d. h. String ist nach Ablauf der Funktion ganz leer. 
// Irene Schmidtmann 14.12.1995 13:50 
 
 
BOOL txt_trim_r (char * src)
{ 
 int i,n;

 if ((n = strlen (src)) <= 0)
  return (FALSE);

 for (i = n - 1;i >= 0 && src [i] == ' ';i--)
  ;

 src [i + 1] = '\0';

 return (TRUE);
}

/******************************************************************************/
/** convert german umlauts
 * @param src pointer to source string, contains converted string after invocation
 */
 
BOOL txt_cvt_uml (char * src)
{
 char r [SYS_STR_MAX];
 int l,i,j;

 if (src == NULL)
  return (FALSE);

 l = strlen (src);
 j = 0;

 for (i = 0;i < l;i++)
 {
  char c;

  c = toupper (src [i]);

  switch (c)
  {
   case '': strcpy (r + j,"SS"); j += 2; break; 
   case '': strcpy (r + j,"AE"); j += 2; break; 
   case '': strcpy (r + j,"OE"); j += 2; break;
   case '': strcpy (r + j,"UE"); j += 2; break; 
   case '': strcpy (r + j,"AE"); j += 2; break; 
   case '': strcpy (r + j,"OE"); j += 2; break; 
   case '': strcpy (r + j,"UE"); j += 2; break; 
   default:  r [j++] = src [i];           break; 
  }

 } 

 r [j] = '\0';
 strcpy (src,r);

 return (TRUE);
}


/******************************************************************************/
/** reduces multiple separators between two syllables to maximal one separator
 * @param s string to source string                
 */ 

//void proc_trennzeichen ( char *s ) 
 
BOOL txt_cvt_del (char * src)
{ 
  char *p1, *p2, *p; 
  char trennzeichen[] = TRENNZEICHEN; 
  int flag=FALSE;        /* TRUE  falls das Zeichen in s vor dem aktuellen 
                                  ein Trennzeichen ist 
                            FALSE sonst */ 
 
  p2 = src;

  /* Prfen, ob das erste Zeichen in s ein Trennzeichen ist */ 

  for (p = trennzeichen;*p != '\0' && *p2 != *p;p++) 
   ; 

  if (*p != '\0')
  {
    flag = TRUE; 
  } 
 
  for (p1 = src + 1,p2 = src;*p1 != '\0';p1++)
  {
   for (p = trennzeichen;*p!= '\0' && *p1 != *p;p++) 
    ; 
   if (*p == '\0'){          /* *p1 ist kein Trennzeichen */ 
      p2++; 
      *p2 = *p1; 
      flag = FALSE; 
    } 
    else{ 
      if (flag == FALSE){ 
        p2++; 
        *p2 = *p1; 
      } 
      flag = TRUE; 
    } 
  } 
  p2++; 
  *p2 = '\0'; 

 return (TRUE);
} 
 
/******************************************************************************/
/** decipher string
 * @param src pointer to source string
 * @param s1 pointer to string
 * @param s2 pointer to string
 * @param s3 pointer to string
 */
 
BOOL txt_dec (char * src,BYTE dec,char * s1,char * s2,char * s3)
{
 BOOL r;

 if (src == NULL || s1 == NULL || s2 == NULL || s3 == NULL)
  return (FALSE);

 switch (dec)
 {
  case TXT_DEC_N: r = txt_dec_n (src,s1,s2,s3); break;
  case TXT_DEC_F: r = txt_dec_f (src,s1,s2,s3); break;
  default:        r = FALSE; break;
 }

 return (r);
}

/******************************************************************************/
/** construct
 * @param txt
 * @param rgs pointer to registry string
 * @param def pointer to default configuration string
 * @param dic pointerto dictionary information
 */
 
BOOL txt_construct (PSX_STR txt,const char * rgs,const char * def,PSX_DIC * dic)
{
 char s [4000],t [4000];
 int i,j;

 if (txt == NULL)
  return (FALSE);

 if (rgs)
 {
  if (!psx_cfg_get (rgs,s))       // get configuration
  {
   if (def == NULL)
    return (FALSE);
   strcpy (s,def);                // get default configuration string
  }
 }
 else
 {
  if (def == NULL)
   return (FALSE);
  strcpy (s,def);
 }

 //////////

 i = 0;
 j = 0;

 while (s [i] != '\0')
 {
  int k;

  if (s [i] != '$')
  {
   t [j++] = s [i++];
   continue;
  }

  i++;

  for (k = 0;k < dic -> n;k++)
  {
   char * p;
   int    n;

   if (!dic_get_atr (dic,k,&p))
    continue;

   if (p == NULL)
    continue;

   if ((n = strlen (p)) == 0)
    continue;

   if (strncmp (s + i,p,n) != 0)
    continue;

   if (!dic_get_val (dic,k,&p))
    break;

   if (p == NULL)
    break;

   i += n;
   n = strlen (p);
   memcpy (t + j,p,n);
   j += n;

//printf ("Content-type: text/html\n\nt='%s',n=%d<br>\n",t,n);

   break;
  }

 }

 t [j] = 0;

 strcpy (txt,t);

 return (TRUE);
}

/******************************************************************************/
/** check whether c is a delimiter
 * @param c character to be checked
 */
 
BOOL txt_isNameDel (char c)
{
 if (strchr (TRENNZEICHEN,c) == NULL)
  return (FALSE);
  
 return (TRUE);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


