/******************************************************************************/
/* psx-wui.c                                               Web User Interface */
/******************************************************************************/
/** @file psx-wui.c Web User Interface - Source Code File
 * Functions and definitions providing the Web User Interface. The Interface
 * invokes the creation of hypertext documents for the Graphical User Interface.
 * For this functionality the interface uses functions of the TUI module.
 */
 
#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/

#include "psx-wui.inl"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** WUI X Context */
typedef struct _WUI_XCX
{
 BYTE      ctx;           /**< context         */
 const char    * sym;           /**< name of context */
} WUI_XCX;

static const WUI_XCX wui_xcx [] =
{
 { WUI_CTX_SYS,"sys" },     /* system  */
 { WUI_CTX_PTL,"ptl" },     /* portal  */
 { WUI_CTX_REQ,"req" },     /* request */
 { WUI_CTX_ERR,"err" }      /* error   */
};


/** WUI command structure                    */
static const struct
{
 BYTE   cmd;            /**< command         */
 const char * sym;            /**< name of command */
} wui_cmd [] =
{
 { WUI_CMD_RETURN,"Return" },
 { WUI_CMD_RESET ,"Reset"  },
 { WUI_CMD_OK    ,"OK"  },
 { WUI_CMD_CANCEL,"Cancel"  },
 { WUI_CMD_HISTORY,"History"  }
};

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** get web user interface context (default: portal) and store in xcx
 * @param xcx pointer to WUI context, result if invokation
 */
 
static BOOL wui_xcx_get (WUI_XCX * xcx)
{
 IDX i;
 NUM n;
 char * t;

 if (xcx == NULL)
  return (FALSE);

 memset (xcx,0,sizeof (*xcx));

 if (!cgi_get ("ctx",&t)) // get context, default: portal
  t = "ptl";

 n = sizeof (wui_xcx) / sizeof (wui_xcx [0]);

 for (i = 0;i < n;i++)
 {
  if (strcmp (wui_xcx [i].sym,t) != 0)
   continue;

  *xcx = wui_xcx [i];                // get context
  return (TRUE);
 }

 return (FALSE);  // no valid context found
}

/******************************************************************************/
/** get command
 * @param cmd pointer to command byte
 */
 
static BOOL wui_cmd_get (BYTE * cmd)
{
 IDX i;
 NUM n;
 char * t;

 if (cmd == NULL)
  return (FALSE);

 *cmd = WUI_CMD_NIL; 

 if (!cgi_get ("cmd",&t))      /* get command from cgi                        */
  t = "ptl";                   /* if no command is found, display portal page */

 n = sizeof (wui_cmd) / sizeof (wui_cmd [0]);       /* get size of cmd struct */

 for (i = 0;i < n;i++)
 {
  if (strcmp (wui_cmd [i].sym,t) != 0)
   continue;

  *cmd = wui_cmd [i].cmd;
  return (TRUE);
 }

 return (FALSE);
}

/******************************************************************************/
/** put version information into dictionary
 * @param wui pointer to web user interface
 */
  
static BOOL wui_put_dic (WUI * wui)
{
 SYS_STR num,tsp,bld;

 if (wui == NULL)
  return (FALSE);

 ver_get (num,tsp,bld);

 dic_put_new (wui -> dic,"PSX_SYM","PSX");             // add new dic item
 dic_put_new (wui -> dic,"PSX_NAME","PID-Generator");
 dic_put_new (wui -> dic,"VER_NUM",num);
 dic_put_new (wui -> dic,"VER_TSP",tsp);
 dic_put_new (wui -> dic,"VER_BLD",bld);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
BOOL wui_get (WUI * wui);   /* necessary prototype */
/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/******************************************************************************/
/** initialize web user interface
 */
 
BOOL wui_init (WUI * wui)
{
 if (wui == NULL)
  return (FALSE);

 memset (wui,0,sizeof (*wui));

 if (!dic_create (&wui -> env))     /* create environment dictionary */
  return (FALSE);

 wui -> sch = NULL;

 if (!fsp_create (&wui -> fmt))     /* create format specification   */
  return (FALSE);

 if (!dic_create (&wui -> dic))     /* create dictionary             */
  return (FALSE);

 if (!wui_get (wui))                /* get information               */
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** exit web user interface
 * @param wui pointer to webuser interface structure
 */
 
BOOL wui_exit (WUI * wui)
{
 if (wui == NULL)
  return (FALSE);

 if (wui -> env)
  dic_delete (&wui -> env);

 if (wui -> dic)
  dic_delete (&wui -> dic);

 return (TRUE);
}

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/******************************************************************************/
/** get information (from cgi query string) like context, status, command and other parameters
 * @param wui WUI pointer, contains web user inferface information after invocation
 */
 
BOOL wui_get (WUI * wui)
{
 WUI_XCX xcx;
 char * t;
 IDX i;

 if (wui == NULL)
  return (FALSE);

 if (!wui_xcx_get (&xcx))
  return (FALSE);

 wui -> ctx = xcx.ctx;

 if (cgi_get ("sts",&t))
  wui -> sts = atoi (t);
 else
  wui -> sts = 0;

 wui_cmd_get (&wui -> cmd); // e.g. "Return", "Reset", "OK", ...
 
 // for each cgi parameter: put into wui.env struct
 for (i = 0;i < cgi.par_num;i++)
  if (!dic_put (wui -> env,cgi.par [i].sym,cgi.par [i].val))
   return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/******************************************************************************/
/** invoke tui output
 * @param wui pointer to web user interface structure
 */
 
BOOL wui_put (WUI * wui)
{
 TUI_CTX tcx;      //template user interface context
 BOOL r = TRUE;

 if (wui == NULL)
  return (FALSE);

 wui_put_dic (wui);

 memset (&tcx,0,sizeof (tcx));

 tcx.tpl = wui -> tpl;
 tcx.sch = wui -> sch;
 tcx.dic = wui -> dic;
 tcx.dbg = wui -> dbg;
 if (wui -> tgt)
  strcpy (tcx.tgt,wui -> tgt);
 else
  strcpy (tcx.tgt,"");

 if (!tui_put (&tcx))
  r = FALSE;

 return (r);
}

/******************************************************************************/
/******************************************************************************/
/******************************************************************************/



