/******************************************************************************/
/* psx.c                                                      Pseudonymizator */
/******************************************************************************/
/** @file psx.c Pseudonymisator - Source Code File
 * This file includes the main-function, the main instantiation of the PSX
 * datastructure, several procedures for logging and output (HTML and Console)
 * and test functions.
 */
 

#include "psx.h"

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** default registry path
 */
 
#define PSX_RGS_PTH "psx.cfg"           

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** Main instantiation.
 * Holds configuration as obtained from configuration file (path to logfile,
 * database name etc.).
 */
 
PSX psx;                                

/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/**
 * check if program is called via cgi
 *
 * @return TRUE if program was called via cgi
 * @return FALSE if program was not called via cgi
 *
 * The return value is assigned to psx.cgi. psx.dbg is set
 * to true if cgi mode is set and debug mode is enabled.
 */
 
static BOOL psx_cgi ()
{
 PSX_STR p1,p2,t;
 char * s;
 BOOL r = FALSE;
 int i;

 if (!sci_par_get (1,p1))
  return (FALSE);

 if (strcmp (p1,"gui") == 0)
  r = TRUE;

 psx.cgi = r;
 psx.dbg = 0;

 if (psx.cgi)
 {   
  cgi_init ();

  if (cgi_get ("dbg",&s))         /* dbg: enable debugging */
   if (strcmp (s,"1") == 0)
    psx.dbg = TRUE;
 }

 return (r);
}


/******************************************************************************/
/* Private                                                                    */
/******************************************************************************/
/** print banner if not in cgi mode to STDERR
 */

static void pss_bnr (void)
{
 SYS_STR num,tsp,bld;
 FILE * f = stderr;

 if (psx_cgi ())                        /* no banner in cgi mode              */
  return;

 ver_get (num,tsp,bld);                 /* retrieve version information       */

 fprintf (f,"\n");
 fprintf (f,"PID-Generator %s [%s]\n",num,bld);
 fprintf (f,"[P] 2000-2003 M. Wagner\n");
 fprintf (f,"\n");
}

/******************************************************************************/
/** get information about registry file
 * @param pth path to registry file
 * @return TRUE if information could be retrieved i.e if registry was found
 * @return FALSE if information could not be retrieved
 */
 
static BOOL psx_rgs_locate (PSX_STR pth)
{
 int r;
 struct stat s;  // struct for Inode information

 if (pth == NULL)	// kein Pfad gefunden
  return (FALSE);

 r = stat (pth,&s);	// stat(): puts Inode information of file "pth" into struct s

 if (r != 0)
  return (FALSE);

 return (TRUE);
}

/******************************************************************************/
/** check if moule should be logged
 * @param mod string pointer to name of module
 * @return TRUE if module should be logged
 * @return FALSE if module should not be logged
 */

static BOOL psx_log_match (const char * mod)
{
 char msk [1000],t [1000];
 int i;

 if (mod == NULL)                       /* no module, no log  */
  return (FALSE);

 if (!psx_cfg_get (PSX_CFG_LGM,msk))    /* retrieve log mask  */
  return (FALSE);

 str_trim (msk);

 if (strcmp (msk,"-") == 0)
  return (FALSE);

 i = 0;                                 /* search for matching module */

 while (str_tok (msk,&i,":",t))         /* copy text up to next colon into t */
  if ((strcmp (t,mod) == 0) || (strcmp (t,"*") == 0)) /* module name oder '*'  in msk */
   return (TRUE);

 return (FALSE);
}

/******************************************************************************/
/* Public                                                                     */
/******************************************************************************/
/** retrieve registry item
 * @param key string pointer to name of registry item
 * @param val string pointer, contains value of registry item after invocation
 * @return TRUE if value could be retrieved
 * @return FALSE if value could not be retrieved
 */

BOOL psx_cfg_get (const char * key,char * val)
{
 PSX_STR t;

 if (key == NULL || val == NULL)
  return (FALSE);

 if (!rgs_get (psx.rgs,key,t)) 	// get registry item
 {
  strcpy (val,"");
  return (FALSE);
 }

 t [PSX_STR_MAX - 1] = 0;

 strcpy (val,t);

 return (TRUE);
}

/******************************************************************************/
/** print HTML formatted line
 * @param f pointer to format
 * @param l variable list of arguments
 */
 
static BOOL psx_print_l (const char * f,va_list l)
{
 if (psx.cgi)         /* html output */
 {
  cgi_put ("<table width='100%' bgcolor='#AA0000'>\n");
  cgi_put ("<tr>\n");
  cgi_put ("<td>\n");
  cgi_put ("<font face='Courier' size='2' color='#FFFFFF'>\n");
  cgi_put_l (f,l);
  cgi_put ("</font>\n");
  cgi_put ("</td>\n");
  cgi_put ("</tr>\n");
  cgi_put ("</table>\n");
 }
 else
 {
  sys_put_l (f,l);    /* print to command line */
 }

 return (TRUE);
}

/******************************************************************************/
/** print message to stdout, can be used like printf()
 * @param fmt pointer to format string
 * @param ... variable argument list
 */
 
BOOL psx_print (const char * fmt,...)
{
 BOOL r = TRUE;
 va_list l;

 if (fmt == NULL)
  return (FALSE);

 va_start (l,fmt);
 r = psx_print_l (fmt,l);
 va_end (l);

 return (r);
}

/******************************************************************************/


static BOOL psx_put_cgi (const char * f,va_list l)
{
 if (!psx.cgi)
  return (FALSE);

 cgi_put ("<table width='100%' bgcolor='#000000'>\n");
 cgi_put ("<tr>\n");
 cgi_put ("<td>X \n");
 cgi_put ("<font face='Courier' size='2' color='#FFFFFF'>\n");
 cgi_put_l (f,l);
 cgi_put ("</font>\n");
 cgi_put ("</td>\n");
 cgi_put ("</tr>\n");
 cgi_put ("</table>\n");

 return (TRUE);
}


static BOOL psx_put_l (const char * f,va_list l)
{
 if (psx.cgi)
 {
  //psx_put_cgi (f,l); 
 }
 else
 {
  char t [10000];

  vsprintf (t,f,l);
  sys_put ("> %s",t);
 }

 return (TRUE);
}

/******************************************************************************/
/** write to console, can be used like printf()
 * @param fmt pointer to format string
 * @param ... variable argument list
 */

BOOL psx_put (const char * fmt,...)
{
 BOOL r = TRUE;
 va_list l;

 va_start (l,fmt);
 //r = psx_put_l (fmt,l);
 psx_print ("%>%<","> ");
 psx_print_l (fmt,l);
 va_end (l);

 return (r);
}

/******************************************************************************/
/** write verbose
 * @param fmt pointer to format string
 * @param ... variable arguments
 */

BOOL psx_put_v (const char * fmt,...)
{
 SYS_STR t;
 BOOL r = TRUE;
 va_list l;

 if (!psx.cfg.verbose)   /* no verbose flag, no output */
  return (FALSE);

 va_start (l,fmt);
 r = psx_put_l (fmt,l);
 va_end (l);

 return (r);
}

/******************************************************************************/
/** log with argument list
 * @param fmt pointer to format string
 * @param l variable argument list 
 */

BOOL psx_log_l (const char * fmt,va_list l)
{
 if (psx.log == NULL || fmt == NULL)
  return (FALSE);

 sys_log_lf (psx.log,fmt,l);

 /*
 if (psx.cgi && psx.dbg)
 {
   psx_put_cgi (fmt,l);   // send html-formatted contents of logfile to cgi stream
  }
*/

 return (TRUE);
}

/******************************************************************************/
/** log for module mod
 * @param mod string pointer to module name
 * @param fmt pointer to format string
 * @param ... variable arguments
 */

BOOL psx_log_m (const char * mod,const char * fmt,...)
{
 char f [10000],m [1000], * p;
 int i;
 va_list l;

 if (psx.log == NULL || mod == NULL || fmt == NULL)
  return (FALSE);

 if (!psx_log_match (mod)) // module selected for logging (in registry item "msk")
  return (FALSE);

 strcpy (f,mod);
 strcat (f,": ");
 strcat (f,fmt); 

 va_start (l,fmt);
 psx_log_l (f,l);
 va_end (l);

 return (TRUE);
}

/******************************************************************************/
/** do formatted logging
 * @param fmt pointer to format string
 * @param ... variable arguements
 */
 
BOOL psx_log (const char * fmt,...)
{
 va_list l;

 if (psx.log == NULL || fmt == NULL)
  return (FALSE);

 va_start (l,fmt);
 psx_log_l (fmt,l);
 va_end (l);

 return (TRUE);
}

/******************************************************************************/
/** get text from log file
 * @param txt pointer to text after invocation
 */
 
BOOL psx_log_get (char ** txt)
{
 FILE * f;
 PSX_STR t;
 NUM s,n;

 if (txt == NULL)
  return (FALSE);

 if (psx.log == NULL)
  return (FALSE);

 if (!psx_cfg_get (PSX_CFG_LGF,t))
  return (FALSE);

 /*************************************/

 fflush (psx.log);

 *txt = NULL;

 if ((f = fopen (t,"rt")) == NULL)
  return (FALSE);

 if (!fio_getsize (f,&s))
 {
  fclose (f);
  return (FALSE);
 }

 n = s - psx.logpos;

 fseek (f,psx.logpos,SEEK_SET);

 if (feof (f))
 {
  fclose (f);
  return (FALSE);
 }

 if (((*txt) = (char *) malloc ((n + 1) * sizeof (char))) == NULL)
 {
  fclose (f);
  return (FALSE);
 }

 fread (*txt,n,1,f);
 (*txt) [n] = 0;

 return (TRUE);
}

/******************************************************************************/
/** initialize system
 * @param ac number of arguments
 * @param av string pointer to arguments
 * @return TRUE if initiatian was successful
 * @return FALSE if initiation failed
 */

static int psx_init (int ac,char ** av)
{
 char p [1000],t [1000];		/*  p: path to configfile */
                            /*  t: path to logfile    */

 memset (&psx,0,sizeof (psx));

 if (sizeof (DWORD) != 4)		/* no 32 bit system       */
 {
  psx_put ("system word size error");
  exit (-1);
 }

 sys_init (ac,av);          /* set start values of system */
 srand (time (NULL));
 memset (&psx,0,sizeof (psx));

 psx_cgi ();				// check if program is called via cgi

 pss_bnr ();				// print banner if not in cgi mode
 
 /* get path to config file or default value as defined in PSX_RGS_PTH */
 sci_opt_get ("c:s",p,PSX_RGS_PTH);	

 if (!psx_rgs_locate (p))     /* configuration file not found */
 {
  PSX_STR t;

  if (sci_par_peek (t))       /* get command name */

  /* if command is "help", do not print error message, execute help command instead */
   if (*t== 'h')
   {
    int r = 0;

    sci_prc (psx_int,&r);     /* print help page */
    return (FALSE);
   }

  psx_put ("missing configuration");

  return (FALSE);
 }

 sci_opt_get ("v:b",&psx.cfg.verbose,0); /* get verbose mode; default 0     */

 psx_put_v ("loading registry [%s]",p);

 if (!rgs_create (&psx.rgs))
  return (FALSE);

 if (!rgs_load (psx.rgs,p))  /* read configuration file p and store values in psx.rgs */
 {
  psx_put_v ("registry load error [%s]",p);
 }

 pdi_cfg (NULL);             /* configure pdi */

 if (psx_cfg_get (PSX_CFG_LGF,t))
  if ((psx.log = fopen (t,"at")) == NULL)
  {
   psx_put ("log file open error (%s)",t);
   psx.logpos = 0;
  }
  else
  {
   psx.logpos = ftell (psx.log);
  }

 cpi_set_key ("");

 return (TRUE);
}

/******************************************************************************/
/** clean up system
 */
 
static int psx_exit (void)
{
 sys_exit ();

 rgs_delete (&psx.rgs);

 if (psx.log)
 {
  fclose (psx.log);
  psx.log = NULL;
 }

 return (0);
}

/******************************************************************************/
/** execute
 */
 
static int psx_run (void)
{
 int r = 0;
  
 LOG (PSX_MOD_SYS,"start");

 if (!psx_cfg_get ("aes.key",psx.key))
 {
  psx_put ("missing AES key");
  return (-1);
 }

 cti_init ();           /* initialize AES structure */
 aes_init ();           /* generate tables for encryption */

 sci_prc (psx_int,&r);  /* call function; psx_int is extern */
 aes_exit ();
 cti_exit ();           /* clean and quit AES structure */

 LOG (PSX_MOD_SYS,"exit (%d)",r);
 
 return (r);
}

/******************************************************************************/
/* Main                                                                       */
/******************************************************************************/

void test_stk ()
{
 PSX_STK * stk;
 char * s1 = "Test 1",* s2 = "Test 2",* s3 = "Test 3",* s;
 //char s1[] = "Test 1", s2[] = "Test 2", s3[] = "Test 3",* s;
 typedef struct _FOO { int x; int y; int z; } FOO;
 FOO f1,f2,f3,f;

 printf ("> create\n");

 if (!stk_create (&stk))
  exit (-1);

 printf ("> push\n");
 stk_push (stk,&s1);
 printf ("> push\n");
 stk_push (stk,&s2);
 printf ("> push\n");
 stk_push (stk,&s3);

 stk_pop (stk,&s); printf ("> pop: %s\n",s);
 stk_pop (stk,&s); printf ("> pop: %s\n",s);
 stk_pop (stk,&s); printf ("> pop: %s\n",s);

 stk_set_esz (stk,sizeof (FOO));

 f1.x = 1;f1.y = 2;f1.z = 3;
 f2.x = 4;f2.y = 5;f2.z = 6;
 f3.x = 7;f3.y = 8;f3.z = 9;

 printf ("> push\n");
 stk_push (stk,&f1);
 printf ("> push\n");
 stk_push (stk,&f2);
 printf ("> push\n");
 stk_push (stk,&f3);

 stk_pop (stk,&f); printf ("> pop: [%d,%d,%d]\n",f.x,f.y,f.z);
 stk_pop (stk,&f); printf ("> pop: [%d,%d,%d]\n",f.x,f.y,f.z);
 stk_pop (stk,&f); printf ("> pop: [%d,%d,%d]\n",f.x,f.y,f.z);

 if (!stk_delete (&stk))
  exit (-1);

 printf ("> OK\n");
}

/******************************************************************************/
/** Main
 *  @param ac number of arguments
 *  @param av string pointer to arguments
 */
 
int main (int ac,char ** av)
{
 int r = -1;

 if (psx_init (ac,av))
  r = psx_run ();
  
 psx_exit ();

 return (r);
}


/******************************************************************************/
/******************************************************************************/
/******************************************************************************/


